<?php

namespace Mtc\Stripe;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\PaymentForm;
use Mtc\Modules\Members\Classes\Auth;
use Mtc\Modules\Members\Classes\Member;
use Stripe\Charge;
use Stripe\Error\Card;
use Stripe\Error\InvalidRequest;

/**
 * Stripe Payment Gateway
 *
 * @package  Mtc\Stripe
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Stripe implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('stripe.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        return new PaymentForm('stripe-payment', 'vue-component', [
            'stripe_public_key' => config('stripe.public_key'),
            'invoice_id' => $invoice->getId(),
            'name' => __('stripe::stripe.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        //Set Api Key with Stripe library
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $charge_data = $this->chargeInvoice($request, $invoice);
        return [
            'provider' => 'stripe',
            'amount' => $charge_data->amount / 100,
            'reference' => $charge_data->id,
            'details' => $charge_data,
            'confirmed_at' => now(),
        ];

    }

    /**
     * Charge Invoice outstanding amount
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @throws Card
     * @throws InvalidRequest
     */
    public function chargeInvoice(Request $request, InvoiceRepositoryContract $invoice)
    {
        $charge = Charge::create([
            'customer' => $this->getCustomerId($request, $invoice->getEmail()),
            'amount' => $invoice->getOutstandingAmount() * 100, // Stripe charges pennies
            'currency' => config('currencies.default_currency'),
            'metadata' => [
                'Order Reference' => 'Invoice ' . $invoice->getId()
            ]
        ]);

        // Throw an exception if we have an invalid payment response
        if (!in_array($charge->outcome->type, config('stripe.allowed_charge_statuses', []))) {
            throw new InvalidRequest(json_encode($charge), false);
        }

        return $charge;
    }

    /**
     * Get the customers stripe id
     *
     * @param Request $request
     * @param string $email
     * @return null|string
     * @throws Card
     */
    protected function getCustomerId(Request $request, $email)
    {
        $token = $request->input('stripe_token');

        if (empty($token)) {
            // Token is not present, we can't process anything
            throw new Card('Card details missing', false, false, false, false, false);
        }

        /** @var Member $user */
        $member = Auth::getLoggedInMember();
        $customer_id = App::make(Customer::class)->findOrCreateCustomerId($member, $email);

        $customer = \Stripe\Customer::retrieve($customer_id);
        $customer->source = $token;
        $customer->save();

        return $customer_id;
    }
}
