<?php

namespace Mtc\Stripe\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;

/**
 * Class StripeController
 * 
 * @package Mtc\Stripe\Http\Controllers
 */
class StripeController
{
    /**
     * 
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice)
    {
        \Stripe\Stripe::setApiKey(config('stripe.private_key'));
        $invoice->load($request->input('invoice_id'));

        if ($request->has('payment_method_id')) {
            // Create payment intent
            $intent = \Stripe\PaymentIntent::create([
                // Amount in pence / cents instead of eur/pounds
                'description' => 'Payment on ' . config('app.name') . ' order ' . $invoice->getReference(),
                'amount' => $invoice->getOutstandingAmountInCurrency() * 100,
                'currency' => $invoice->getCurrency(),
                'payment_method' => request()->input('payment_method_id'),
                'customer' => $request->input('customer_id') !== 'false' ? $request->input('customer_id') : null,
                'confirmation_method' => 'manual',
                'save_payment_method' => $request->input('customer_id') && $request->input('store_card') == 1,
                'confirm' => true,
            ]);
        } elseif (request()->has('payment_intent_id')) {
            // Confirm payment intent
            $intent = \Stripe\PaymentIntent::retrieve(request()->input('payment_intent_id'));
            $intent->confirm();
        }

        if (in_array($intent->status, ['requires_action', 'requires_source_action']) && $intent->next_action->type === 'use_stripe_sdk') {
            // Incomplete attempt, tell the client to handle the action
            return [
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret
            ];
        }

        if ($intent->status === 'succeeded') {
            $invoice->savePayment([
                'provider' => 'stripe',
                'amount' => $intent->amount / 100,
                'reference' => $intent->id,
                'details' => $intent,
                'confirmed_at' => now(),
            ]);
            return [
                'success' => true,
                'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'),
            ];
        }

        return response([
            'error' => 'Invalid PaymentIntent status'
        ], 500);
    }
}
