<?php

namespace Mtc\Stripe\Providers;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Events\LoadExternalPaymentGateways;
use Mtc\Basket\Http\Resources\CheckoutAjax;
use Mtc\Basket\Events\AddExternalPaymentGatewayScripts;
use Mtc\Checkout\Events\AddPaymentGatewayScripts;
use Mtc\Stripe\Console\Commands\InstallStripe;
use Mtc\Stripe\Contracts\StripePaymentRequestBasketFillInterface;
use Mtc\Stripe\Listeners\AddStripeExternalScripts;
use Mtc\Stripe\Listeners\AddStripePaymentRequestButton;
use Mtc\Stripe\Stripe;
use Mtc\Stripe\StripePaymentRequest;

/**
 * Class StripeServiceProvider
 *
 * @package Mtc\Stripe\Providers
 */
class StripeServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'stripe');
        $this->app['payment']->extend('stripe', function($app) {
            return $this->app->make(config('stripe.payment_gateway_class'));
        });

        if (!$this->app->runningInConsole()) {
            Event::listen(AddPaymentGatewayScripts::class, AddStripeExternalScripts::class);

            if (config('stripe.additional_payment_gateways')) {
                $this->bootStripePaymentRequestGateway();
            }
        } else {
            // Publishing assets.
            $this->publishes([
                dirname(__DIR__, 2) . '/resources/js' => resource_path('js/vendor'),
            ], 'assets');
        }

        Route::middleware('web')->group(dirname(__DIR__, 2) . '/routes/web.php');
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/stripe.php', 'stripe');
    }

    /**
     * Boot the logic specific to payment request flow
     */
    protected function bootStripePaymentRequestGateway()
    {
        Event::listen(AddExternalPaymentGatewayScripts::class, AddStripeExternalScripts::class);
        Event::listen(LoadExternalPaymentGateways::class, AddStripePaymentRequestButton::class);
        Event::listen(CheckoutAjax::class, function () {
            Event::dispatch(LoadExternalPaymentGateways::class, App::make(BasketRepositoryInterface::class));
        });

        $this->app['payment']->extend('stripe-external', function() {
            return new StripePaymentRequest();
        });

        $this->app->singleton(StripePaymentRequestBasketFillInterface::class, function () {
            return $this->app->make(config('stripe.basket_fill_factory'));
        });
    }
}
