<?php

namespace Mtc\Stripe;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Contracts\HasDeferredCharges;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;
use Mtc\Members\Facades\MemberAuth;
use Mtc\Stripe\Config\StripeConfig;
use Stripe\Error\Card;
use Stripe\PaymentIntent;
use Stripe\Refund;
use Stripe\SetupIntent;

/**
 * Stripe Payment Gateway
 *
 * @package  Mtc\Stripe
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Stripe implements PaymentGateway, HasDeferredCharges, HandlesRefunds
{
    public function __construct(protected StripeConfig $config)
    {
        //
    }

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= $this->config->minimalTransactionAmount()) {
            return false;
        }

        return App::make($this->config->applicableCheckClass())->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        \Stripe\Stripe::setApiKey($this->config->privateKey());
        if (config('checkout.deferred_payments')) {
            $setup_intent = \Stripe\SetupIntent::create([
                'usage' => 'off_session',
            ]);

            return new PaymentForm('stripe-payment', 'vue-component', [
                'stripe_public_key' => $this->config->publicKey(),
                'invoice_id' => $invoice->getId(),
                'name' => __('stripe::stripe.payment_option_name'),
                'customer_cards' => $this->getCustomerCards(),
                'can_save_card' => $this->canSaveCards(),
                'customer' => $this->getCustomerToken(),
                'deferred_payments' => true,
                'intent_client_secret' => $setup_intent->client_secret,
            ]);
        }

        return new PaymentForm('stripe-payment', 'vue-component', [
            'stripe_public_key' => $this->config->publicKey(),
            'invoice_id' => $invoice->getId(),
            'name' => __('stripe::stripe.payment_option_name'),
            'customer_cards' => $this->getCustomerCards(),
            'can_save_card' => $this->canSaveCards(),
            'customer' => $this->getCustomerToken(),
            'deferred_payments' => false,
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        // Regular flow is handled in StripeController, only deferred flow is processed here
        if (config('checkout.deferred_payments') === false) {
            return [];
        }

        \Stripe\Stripe::setApiKey($this->config->privateKey());
        $intent = SetupIntent::retrieve($request->input('setup_intent_id'));
        $customer = \Stripe\Customer::create([
            'payment_method' => $intent->payment_method,
        ]);

        return [
            'provider' => 'stripe',
            'amount' => $invoice->getOutstandingAmount(),
            'currency_code' => $invoice->getCurrency(),
            'amount_in_currency' => $invoice->getOutstandingAmountInCurrency(),
            'reference' => $request->input('setup_intent_id'),
            'details' => [
                'customer_id' => $customer->id,
            ],
            'confirmation_status' => $this->config->confirmationStatus(),
        ];
    }

    /**
     * Check if payment can have a deferred charge
     *
     * @param Payment $payment
     * @return bool
     */
    public function allowDeferredCharge(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true || strpos($payment->reference, 'seti') !== 0) {
            return false;
        }

        try {
            \Stripe\Stripe::setApiKey($this->config->privateKey());
            $intent = SetupIntent::retrieve($payment->reference);
            return $intent->usage === 'off_session' && !empty($payment->details['customer_id']);
        } catch (\Exception $exception) {
            return false;
        }
    }

    /**
     * Charge a payment that was set up as deferred
     *
     * @param Payment $payment
     * @return bool
     * @throws \Exception
     */
    public function chargeDeferredPayment(Payment $payment)
    {
        if (config('checkout.deferred_payments') !== true) {
            return false;
        }

        try {
            \Stripe\Stripe::setApiKey($this->config->privateKey());
            $payment_methods = \Stripe\PaymentMethod::all([
                'customer' => $payment->details['customer_id'],
                'type' => 'card',
            ]);

            $intent = \Stripe\PaymentIntent::create([
                'amount' => round($payment->amount_in_currency * 100),
                'currency' => $payment->currency_code,
                'customer' => $payment->details['customer_id'],
                'payment_method' => $payment_methods->data[0]->id,
                'off_session' => true,
                'confirm' => true,
                'automatic_payment_methods' => [
                    'enabled' => true,
                    'allow_redirects' => 'never',
                ]
            ]);

            $payment->update([
                'reference' => $intent->id,
            ]);

            return true;
        } catch (Card $card_error) {
            $error_data = $card_error->getJsonBody();
            if (!empty($error_data['error']['payment_intent']['charges']['data'][0]['outcome'])) {
                $details = $payment->details;
                $details['error'] = $error_data['error']['payment_intent']['charges']['data'][0]['outcome'];
                $payment->update([
                    'details' => $details
                ]);
                throw new \Exception($details['card_error']['seller_message']);
            }
        } catch (\Exception $exception) {
            throw new \Exception($exception->getMessage());
        }

        if (in_array($intent->status, ['requires_action', 'requires_source_action'])) {
            $payment->triggerRescue();
            throw new \Exception('Card security error, requesting customer to authorise payment');
        }

        throw new \Exception($intent->status);
    }

    /**
     * Check if payment is refundable by payment gateway
     *
     * @param Payment $payment
     * @return bool|array
     */
    public function isRefundable(Payment $payment)
    {
        try {
            \Stripe\Stripe::setApiKey($this->config->privateKey());
            $intent = PaymentIntent::retrieve($payment->reference);
            if ($intent->charges->data[0]->refunds->total_count > 0) {
                $payment->refundable_amount = ($intent->charges->data[0]->amount - $intent->charges->data[0]->amount_refunded) / 100;
            }
            return $intent->charges->data[0]->refunded === false;
        } catch (\Exception $exception) {
            return false;
        }
    }

    /**
     * Process a refund on payment
     *
     * @param Payment $payment
     * @param null $amount
     * @return bool|array
     */
    public function refund(Payment $payment, $amount = null)
    {
        \Stripe\Stripe::setApiKey($this->config->privateKey());
        $refund = \Stripe\Refund::create([
            'amount' => $amount ? (int)($amount * 100) : null,
            'payment_intent' => $payment->reference,
        ]);

        if ($refund->status !== 'succeeded') {
            return false;
        }

        return [
            'reference' => $refund->balance_transaction,
            'amount' => $refund->amount / 100,
        ];
    }

    /**
     * Create customer record
     *
     * @param $payment_method_id
     * @param $email
     * @param bool $store_card
     * @return string
     */
    public function createCustomer($payment_method_id, $email, $store_card = false)
    {
        $member = MemberAuth::user();
        $customer = \Stripe\Customer::create([
            'email' => $member->email ?? $email,
            'payment_method' => $payment_method_id,
        ]);

        if ($member && $store_card) {
            $member->tokens()
                ->create([
                    'provider' => 'stripe',
                    'token' => $customer->id,
                ]);
        }

        return $customer->id;
    }

    public function paymentDescription(InvoiceRepositoryContract $invoice): string
    {
        return 'Payment on ' . config('app.name') . ' order ' . $invoice->getReference();
    }

    public function paymentMetadata(InvoiceRepositoryContract $invoice): array
    {
        return [];
    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function canSaveCards()
    {
        return $this->config->cardStorage() && MemberAuth::check();
    }

    /**
     * Find customer saved cards
     * @return mixed
     */
    protected function getCustomerCards()
    {
        if ($this->config->cardStorage() === false) {
            return [];
        }

        $token = $this->getCustomerToken();
        if (!$token) {
            return [];
        }

        //retrieve stripe customer
        $cards = \Stripe\PaymentMethod::all([
            'customer' => $token,
            'type' => 'card'
        ]);

        return collect($cards->values()[1])
            ->map(function ($card) {
                return [
                    'id' => $card->id,
                    'brand' => ucfirst($card->card->brand),
                    'number' => '**** **** **** ' . $card->card->last4,
                    'expiry' => $card->card->exp_month . '/' . $card->card->exp_year
                ];
            });
    }

    /**
     * Get customers stripe token
     *
     * @return bool|string
     */
    protected function getCustomerToken()
    {
        if ($this->config->cardStorage() === false) {
            return false;
        }

        $member = MemberAuth::user();
        if (MemberAuth::check() === false) {
            return false;
        }

        return $member->tokens()->where('provider', 'stripe')->first()->token ?? false;
    }
}
