<?php

namespace Mtc\Stripe\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Stripe\Contracts\AbstractStripeConfig;
use Mtc\Stripe\Stripe;
use Stripe\Card;
use Stripe\PaymentIntent;

/**
 * Class StripeController
 *
 * @package Mtc\Stripe\Http\Controllers
 */
class StripeController
{
    public function __construct(protected readonly AbstractStripeConfig $config)
    {
        //
    }

    /**
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|\Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice)
    {
        \Stripe\Stripe::setApiKey($this->config->privateKey());
        $invoice->load($request->input('invoice_id'));

        if ($invoice->hasConflicts()) {
            return [
                'success' => false,
                'issues' => $invoice->getConflicts(),
                'redirect' => route('checkout.index'),
            ];
        }

        if ($invoice->isPaid()) {
            return [
                'success' => true,
                'invoice_id' => $invoice->getId(),
            ];
        }

        try {
            if ($request->has('payment_method_id')) {
                $name = $this->getBillingName($invoice);
                $customer_id = $this->getPaymentCustomerId($request, $invoice->getEmail(), $name);

                $stripe = app($this->config->paymentGatewayClass());
                // Create payment intent
                $intent = PaymentIntent::create([
                    // Amount in pence / cents instead of eur/pounds
                    'description' => $stripe->paymentDescription($invoice),
                    'amount' => $invoice->getOutstandingAmountInCurrency() * 100,
                    'currency' => $invoice->getCurrency(),
                    'payment_method' => $request->input('payment_method_id'),
                    'customer' => $customer_id ?? null,
                    'save_payment_method' => $this->shouldSavePaymentMethod($request),
                    'confirm' => $this->shouldChargeImmediately($request),
                    'metadata' => $stripe->paymentMetadata($invoice),
                    'automatic_payment_methods' => [
                        'enabled' => true,
                        'allow_redirects' => 'never',
                    ]
                ]);
            } elseif (request()->has('payment_intent_id')) {
                // Confirm payment intent
                $intent = PaymentIntent::retrieve(request()->input('payment_intent_id'));
                if ($intent->status !== 'succeeded' && $this->shouldChargeImmediately($request)) {
                    $intent->confirm();
                }
            }
        } catch (Card $exception) {
            return response([
                'error' => $exception->getMessage()
            ], 422);
        }

        if (
            in_array($intent->status, ['requires_action', 'requires_source_action'])
            && $intent->next_action->type === 'use_stripe_sdk'
        ) {
            // Incomplete attempt, tell the client to handle the action
            return [
                'requires_action' => true,
                'payment_intent_client_secret' => $intent->client_secret
            ];
        }

        if ($intent->status === 'succeeded') {
            $invoice->savePayment([
                'provider' => 'stripe',
                'amount' => $intent->amount / 100,
                'reference' => $intent->id,
                'details' => $intent,
                'confirmed_at' => $this->shouldChargeImmediately($request) ? now() : null,
                'confirmation_status' => $this->config->confirmationStatus(),
            ]);

            return [
                'success' => true,
                'invoice_id' => $invoice->getId(),
            ];
        }

        return response([
            'error' => 'Invalid PaymentIntent status'
        ], 500);
    }

    /**
     * Check if card should be stored for the payment method
     *
     * @param $request
     * @return bool
     */
    protected function shouldSavePaymentMethod($request)
    {
        return $request->input('store_card') == 1;
    }

    /**
     * Get the customer ID for the order
     * @param $request
     * @return string|null
     */
    protected function getPaymentCustomerId($request, $email, ?string $name)
    {
        if ($request->input('payment_method_id') && $request->filled('customer_id')) {
            return $request->input('customer_id');
        }

        if ($this->shouldSavePaymentMethod($request) === false) {
            return null;
        }

        return $request->input('customer_id') !== 'false'
            ? $request->input('customer_id')
            : App::make(Stripe::class)->createCustomer(
                $request->input('payment_method_id'),
                $email,
                $request->input('store_card'),
                $name
            );
    }

    /**
     * Check if payment should be charged immediately
     *
     * @param $request
     * @return bool
     */
    protected function shouldChargeImmediately($request)
    {
        return config('checkout.deferred_payments') !== true || $request->input('strict') === 'true';
    }

    private function getBillingName(InvoiceRepositoryContract $invoice): ?string
    {
        $billingAddress = $invoice->getModel()->addresses()->where('type', 'recipient')->first();
        if (!$billingAddress) {
            return null;
        }

        return $billingAddress->first_name . ' ' . $billingAddress->last_name;
    }
}
