<?php

namespace Mtc\Taxonomies\Fields;

use Illuminate\Support\Facades\App;
use Mtc\CustomFields\Contracts\CustomField;
use Mtc\CustomFields\Contracts\CustomFieldValue;
use Mtc\Input\Contracts\FormField;
use Mtc\Taxonomies\Contracts\TaxonomyModel;
use Mtc\Taxonomies\Facades\Taxonomies;

class TaxonomyField extends FormField
{
    protected static $field_options = [];

    public function name(): string
    {
        return __('taxonomies::taxonomies.taxonomy');
    }

    public function uiComponent(): string
    {
        return 'admin-input-select';
    }

    public function hasOptions(): bool
    {
        return true;
    }

    public function fields(): array
    {
        return [
            'taxonomy' => [
                'type' => 'select',
                'name' => 'Taxonomy',
                'instructions' => 'Select taxonomy whose options to display',
                'placeholder' => 'Choose Taxonomy',
                'values' => $this->rootLevelTaxonomies(),
            ],
            'instructions' => [
                'type' => 'textarea',
                'name' => 'Instructions',
                'instructions' => 'Instructions about field. Shown on form when editing record',
            ],
            'required' => [
                'type' => 'checkbox',
                'name' => 'Required?',
            ],
            'default_value' => [
                'type' => 'text',
                'name' => 'Default Value',
                'instructions' => 'Appears when creating a new record',
            ],
            'placeholder' => [
                'type' => 'text',
                'name' => 'Placeholder',
                'instructions' => 'Appears within the input',
            ],
            'prepend' => [
                'type' => 'text',
                'name' => 'Prepend',
                'instructions' => 'Appears before the input',
            ],
            'append' => [
                'type' => 'text',
                'name' => 'Append',
                'instructions' => 'Appears after the input',
            ],
        ];
    }

    /**
     * Get value
     * @param CustomFieldValue $value
     * @return int
     */
    public function getValue(CustomFieldValue $value)
    {
        return (int)$value->value;
    }

    /**
     * Prepare field to be displayed based on config
     *
     * @param mixed $field
     */
    public function prepareForDisplay($field = null)
    {
        if ($field === null) {
            return;
        }

        $field->choices = $this->displayChoices($field->data['taxonomy']);
    }

    public function displayChoices($details = null)
    {
        if (empty(self::$field_options[$details])) {
            self::$field_options[$details] = Taxonomies::ancestorQuery($details)
                ->select('id', 'name')
                ->get();
        };

        return self::$field_options[$details];
    }

    /**
     * Find Root level taxonomies
     * @return mixed
     */
    private function rootLevelTaxonomies()
    {
        return App::make(TaxonomyModel::class)
            ->newQuery()
            ->whereNull('parent_id')
            ->pluck('name', 'id');
    }

    public function validationRules(CustomField $field, bool $is_filled = true): array
    {
        return [
            !empty($field->data['required']) ? 'required' : '',
        ];
    }
}
