<?php

namespace Mtc\Taxonomies\Http\Controllers;

use Mtc\Taxonomies\Jobs\RunTaxonomyImport;
use Illuminate\Foundation\Bus\DispatchesJobs;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Mtc\Breadcrumbs\Facades\Breadcrumbs;
use Mtc\Taxonomies\Facades\Taxonomies;
use Mtc\Taxonomies\Contracts\TaxonomyModel;
use Mtc\Taxonomies\Imports\TaxonomyImport;

class ImportTaxonomyController
{
    use ValidatesRequests;
    use DispatchesJobs;

    public function import(TaxonomyModel $taxonomy_model, $taxonomy)
    {
        $taxonomy_model = $taxonomy_model->findFromSlug($taxonomy);
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));
        Breadcrumbs::addCrumb($taxonomy_model->name, route('taxonomies.show', $taxonomy));
        Breadcrumbs::addCrumb('Import ' . Str::plural($taxonomy_model->name), route('taxonomies.import', $taxonomy));
        return view('taxonomies::import', [
            'taxonomy' => $taxonomy_model,
            'step' => 1,
        ]);
    }

    public function uploadAndReview(Request $request, TaxonomyModel $taxonomy_model, $taxonomy)
    {
        $taxonomy_model = $taxonomy_model->findFromSlug($taxonomy);
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));
        Breadcrumbs::addCrumb($taxonomy_model->name, route('taxonomies.show', $taxonomy));
        Breadcrumbs::addCrumb('Import ' . Str::plural($taxonomy_model->name), route('taxonomies.import', $taxonomy));

        $import_file_name = 'imports/' . $taxonomy_model->slug . '-import-' . date('U')
            . '.' . $request->file('import_file')->getClientOriginalExtension();
        $file_name = $request->file('import_file')->store($import_file_name);

        Session::put('taxonomy-import-file', $file_name);

        $import_data = (new TaxonomyImport())->toArray($file_name);

        return view('taxonomies::import', [
            'taxonomy' => $taxonomy_model,
            'step' => 2,
            'import_columns' => Taxonomies::exportableColumns($taxonomy_model),
            'data' => collect($import_data[0])->take(5),
            'total_records' => count($import_data[0]),
        ]);
    }

    public function runImport(Request $request, TaxonomyModel $taxonomy_model, $taxonomy)
    {
        $taxonomy = $taxonomy_model->findFromSlug($taxonomy);
        $file_name = Session::get('taxonomy-import-file');
        if (empty($file_name)) {
            Session::flash('error', "Can't find uploaded import file. Please submit import file");
            return redirect(route('taxonomies.import', $taxonomy->slug));
        }

        $job = new RunTaxonomyImport(
            $request->input('import_map'),
            $request->input('set_active'),
            $file_name,
            $taxonomy->id
        );
        $this->dispatch($job);

        Session::flash('success', 'Taxonomy Import Started. Success/Issues will be added to notifications');
        return redirect(route('taxonomies.show', $taxonomy->slug));
    }
}
