<?php

namespace Mtc\Taxonomies\Http\Controllers;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Str;
use Mtc\Breadcrumbs\Facades\Breadcrumbs;
use Mtc\Taxonomies\Facades\Taxonomies;
use Mtc\Taxonomies\Http\Requests\CreateTaxonomy;
use Mtc\Taxonomies\Http\Requests\UpdateTaxonomy;
use Mtc\Taxonomies\Http\Resources\TaxonomyList;
use Mtc\Taxonomies\Contracts\TaxonomyModel;

/**
 * Class TaxonomyController
 *
 * @package Mtc\Taxonomies\Http\Controllers\
 */
class TaxonomyController
{
    use ValidatesRequests;

    /**
     * Display a listing of the resource.
     *
     */
    public function index(Request $request, TaxonomyModel $taxonomy)
    {
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));

        return view('taxonomies::index', [
            'taxonomies' => $taxonomy->newQuery()
                ->whereNull('parent_id')
                ->latest()
                ->paginate($request->input('per_page', 20))
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     */
    public function create(TaxonomyModel $taxonomy)
    {
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.create_taxonomy'), route('taxonomies.create'));

        return view('taxonomies::edit', [
            'taxonomy' => $taxonomy
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     */
    public function store(CreateTaxonomy $request)
    {
        $taxonomy = Taxonomies::create($request->input('name'), Str::slug($request->input('name')), $request->input());

        session()->flash('success', __('taxonomies::taxonomies.created'));
        return $request->input('action') === 'continue_edit'
            ? redirect(route('taxonomies.edit', [$taxonomy->id]))
            : redirect(route('taxonomies.show', [$taxonomy->slug]));
    }

    /**
     * Display the specified resource.
     *
     * @param TaxonomyModel $taxonomy
     */
    public function show(Request $request, TaxonomyModel $taxonomy_model, $taxonomy)
    {
        $taxonomy_model = $taxonomy_model->findFromSlug($taxonomy);
        $taxonomy_model->load('children');

        if ($request->wantsJson()) {
            return new TaxonomyList($taxonomy_model);
        }

        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));
        Breadcrumbs::addCrumb($taxonomy_model->name, route('taxonomies.show', $taxonomy));
        return view('taxonomies::show', [
            'taxonomy' => $taxonomy_model,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     */
    public function edit(TaxonomyModel $taxonomy)
    {
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.manage_taxonomies'), route('taxonomies.index'));
        Breadcrumbs::addCrumb(__('taxonomies::taxonomies.edit_taxonomy'), route('taxonomies.edit', $taxonomy));

        return view('taxonomies::edit', [
            'taxonomy' => $taxonomy,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param TaxonomyModel $taxonomy
     * @return Application|RedirectResponse|Response|Redirector|TaxonomyModel
     */
    public function update(UpdateTaxonomy $request, TaxonomyModel $taxonomy)
    {
        $taxonomy->fill($request->input());
        $taxonomy->active = (bool) $request->input('active');
        $taxonomy->save();

        if ($request->wantsJson()) {
            $taxonomy->is_open = false;
            $taxonomy->edit = false;
            $taxonomy->loadCustomFields($taxonomy->getRoot()->customFieldGroup->id ?? null);
            return $taxonomy;
        }
        session()->flash('success', __('taxonomies::taxonomies.updated'));
        return redirect(route('taxonomies.show', [$taxonomy->id]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param TaxonomyModel $taxonomy
     * @return Response
     */
    public function destroy(Request $request, TaxonomyModel $taxonomy)
    {
        $taxonomy->delete();
        if ($request->wantsJson()) {
            return response('success');
        }
        session()->flash('success', __('taxonomies::taxonomies.removed'));
        return back();
    }

    /**
     * Add a child entry
     *
     * @param Request $request
     * @return TaxonomyList
     */
    public function addChild(Request $request, TaxonomyModel $taxonomy)
    {
        $parent = $taxonomy->newQuery()->find($request->input('parent_id'));
        $parent->children()
            ->create([
                'name' => $request->input('name'),
                'description' => $request->input('description')
            ]);

        $parent->load('children');
        return new TaxonomyList($parent);
    }

    /**
     * List taxonomies based of filter
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function showList(Request $request, TaxonomyModel $taxonomy)
    {
        $taxonomies = $taxonomy->newQuery()->withDepth();

        if ($request->input('exclude', false)) {
            /** @var TaxonomyModel $exclude */
            $exclude = $taxonomy->newQuery()->find($request->input('exclude', false));
            $root = $exclude->getRoot();

            $taxonomies->whereDescendantOf($root)
                ->where('id', '!=', $exclude->id)
                ->whereNotDescendantOf($exclude);
        }

        /*
         * Check if we are restricting the list by depth
         * Location selection only allows root level menu so it needs filtering by depth of 0
         */
        if ($request->input('of_depth', false)) {
            $taxonomies->having('depth', '=', $request->input('of_depth'));
        }

        $taxonomies = $taxonomies->orderBy('_lft')->get();

        /*
         * Reorder needs to have key/value map
         */
        if ($request->input('key_value_pair')) {
            $taxonomies = $taxonomies
                ->keyBy(function ($entry) {
                    return '_' . $entry->id;
                })
                ->map(function ($taxonomy) {
                    return str_repeat('-', $taxonomy->depth) . ' ' . $taxonomy->name;
                });
        }

        if ($request->input('prepend_root')) {
            if ($request->input('key_value_pair')) {
                $taxonomies->put(0, trans('taxonomies::taxonomies.root_level'));
            } else {
                $taxonomies->prepend([
                    'id' => '0',
                    'name' => trans('taxonomies::taxonomies.root_level'),
                ]);
            }
        }

        // Return a prepared response
        // Doing this instead of collection as collection is already sorted and we don't want the order to change
        return response()->json($taxonomies);
    }

    /**
     * Change the parent of a taxonomy
     *
     * @param Request $request
     */
    public function changeParent(Request $request, TaxonomyModel $taxonomy)
    {
        $taxonomy = $taxonomy->newQuery()->find($request->input('taxonomy_id'));

        if ($request->input('new_parent') == 0) {
            $taxonomy->getRoot()->appendNode($taxonomy);
        } else {
            $parent = $taxonomy->newQuery()->find($request->input('new_parent'));
            $parent->appendNode($taxonomy);
        }
    }

    /**
     * Change the order of taxonomies
     *
     * @param Request $request
     */
    public function reorder(Request $request, TaxonomyModel $taxonomy_model)
    {
        foreach ($request->input('taxonomies', []) as $key => $taxonomy_data) {
            $taxonomy = $taxonomy_model->newQuery()->find($taxonomy_data['id']);

            try {
                if (!empty($previous_taxonomy)) {
                    $taxonomy->afterNode($previous_taxonomy)->save();
                }
            } catch (\Exception $exception) {
                // Ignore impossible move and proceed
                // e.g. the order of the first child has not changed causes the exception
            }

            // set this taxonomy as previous so we can put next one besides this one
            $previous_taxonomy = $taxonomy;
        }
    }

    /**
     * Change the active state for taxonomy
     *
     * @param TaxonomyModel $taxonomy
     * @return \Illuminate\Contracts\Routing\ResponseFactory|Response
     */
    public function toggleActive(TaxonomyModel $taxonomy)
    {
        $taxonomy->active = !$taxonomy->active;
        $taxonomy->save();
        return response('success');
    }

    public function bulkDelete(Request $request, TaxonomyModel $taxonomy_model)
    {
        $taxonomy_model->newQuery()
            ->whereIn('id', $request->input('taxonomies'))
            ->get()
            ->each(function (TaxonomyModel $taxonomy) {
                $taxonomy->delete();
            });
        return \response('Removed');
    }
}
