<?php

namespace Mtc\Taxonomies;

use Illuminate\Support\Str;
use Mtc\Admin\Facades\Navigation;
use Mtc\CustomFields\Facades\CustomFields;
use Mtc\Taxonomies\Contracts\TaxonomyModel;
use Mtc\Taxonomies\Exceptions\TaxonomyExistsException;

class TaxonomyRepository
{
    /** @var TaxonomyModel */
    protected $model;

    /**
     * TaxonomyRepository constructor.
     * @param TaxonomyModel $taxonomy
     */
    public function __construct(TaxonomyModel $taxonomy)
    {
        $this->model = $taxonomy;
    }

    /**
     * Create a new taxonomy
     */
    public function create(string $name, string $slug, $options = []): TaxonomyModel
    {
        if ($this->exists($slug)) {
            throw new TaxonomyExistsException("{$slug} already exists");
        }

        /** @var TaxonomyModel $taxonomy */
        $taxonomy = $this->model->newQuery()
            ->create([
                'name' => $name,
                'slug' => $slug,
                'tree_structure' => $options['tree_structure'] ?? false,
                'active' => $options['active'] ?? true,
                'max_depth' => $options['max_depth'] ?? null,
            ]);

        if (!empty($options['fields'])) {
            CustomFields::createGroup($taxonomy, $options['fields']);
        }

        $this->addMenuEntry($taxonomy);
        return $taxonomy;
    }

    /**
     * Insert a batch of records to $parent record
     */
    public function insert(TaxonomyModel $parent, array $list_of_records)
    {
        foreach ($list_of_records as $record) {
            /** @var TaxonomyModel $taxonomy */
            $taxonomy = $parent->children()
                ->create([
                    'name' => $record['name'],
                    'active' => $options['active'] ?? true,
                ]);

            if (!empty($record['custom'])) {
                $taxonomy->saveCustomFields($record['custom']);
            }
        }
    }

    /**
     * Check if taxonomy with this slug already exists
     */
    public function exists(string $slug): bool
    {
        return $this->model->newQuery()
            ->where('slug', $slug)
            ->exists();
    }

    /**
     * @param $ancestor_id
     * @return TaxonomyModel
     */
    public function ancestorQuery($ancestor_id)
    {
        return $this->model->newQuery()->whereDescendantOf($ancestor_id);
    }

    public function exportableColumns(TaxonomyModel $taxonomy)
    {
        $fields = collect([
            [
                'id' => 'name',
                'name' => 'Name',
            ],
        ]);
        $taxonomy->loadCustomFields($taxonomy->customFieldGroup->id ?? null);
        collect($taxonomy->custom_fields)
            ->each(function ($field) use ($fields) {
                $fields->push([
                    'id' => $field['slug'],
                    'name' => $field['name'],
                ]);
            });
        return $fields;
    }

    protected function addMenuEntry(TaxonomyModel $taxonomy)
    {
        $navigation = [
            'title' => Str::plural($taxonomy->name),
            'path' => route('taxonomies.show', $taxonomy->slug, false),
            'active_path' => route('taxonomies.show', $taxonomy->slug, false),
        ];

        Navigation::addSecondLevelMenu($navigation, 'taxonomies');
    }
}
