<?php

namespace Mtc\Taxonomies;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Str;
use Kalnoy\Nestedset\NodeTrait;
use Mtc\Taxonomies\Http\Presenters\TaxonomyUrlPresenter;

/**
 * Class Taxonomy
 *
 * @package Mtc\Taxonomies
 */
class Taxonomy extends Model
{
    use NodeTrait;

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'description',
        'image',
        'active',
        'tree_structure',
    ];

    /**
     * @var array
     */
    protected $casts = [
        'tree_structure' => 'boolean',
        'active' => 'boolean',
    ];

    /**
     * @var array
     */
    protected $appends = [
        'url',
    ];

    /**
     * Boot model
     */
    public static function boot()
    {
        parent::boot();

        static::creating(function (self $taxonomy) {
            $taxonomy->ensureUniqueSlug();
        });

        static::addGlobalScope(function ($query) {
            return $query->orderBy('_lft');
        });
    }

    /**
     * URL builder
     *
     * @return TaxonomyUrlPresenter
     */
    public function getUrlAttribute()
    {
        return new TaxonomyUrlPresenter($this);
    }

    /**
     * Make sure that slug for the taxonomy is unique
     */
    protected function ensureUniqueSlug()
    {
        $i = 0;
        $slug_base = $slug = Str::slug($this->name);
        while (self::query()->where('slug', $slug)->exists()) {
            $i++;
            $slug = "{$slug_base}-{$i}";
        }
        $this->slug = $slug;
    }

    /**
     * Find model by slug with fallback of finding by id
     *
     * @param $slug
     * @return Model|null
     */
    public static function find($slug)
    {
        try {
            return self::query()
                ->where('slug', $slug)
                ->firstOrFail();
        } catch (ModelNotFoundException $exception) {
            if (is_numeric($slug)) {
                return self::query()->find($slug);
            }
        }
        return null;
    }

    /**
     * Get the root level taxonomy of this type
     * @return mixed
     */
    public function getRoot()
    {
        return self::query()
            ->whereIsRoot()
            ->where('_lft', '<=', $this->_lft)
            ->where('_rgt', '>=', $this->_rgt)
            ->first();
    }
}
