<?php

namespace Mtc\Taxonomies\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Mtc\Taxonomies\Http\Requests\CreateTaxonomy;
use Mtc\Taxonomies\Http\Requests\UpdateTaxonomy;
use Mtc\Taxonomies\Http\Resources\TaxonomyList;
use Mtc\Taxonomies\Taxonomy;

/**
 * Class TaxonomyController
 *
 * @package Mtc\Taxonomies\Http\Controllers\
 */
class TaxonomyController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = __('taxonomies::taxonomies.manage_taxonomies');
        return template('admin/taxonomies/index.twig', [
            'page_meta' => $this->page_meta,
            'taxonomies' => Taxonomy::query()->whereNull('parent_id')->latest()->paginate($request->input('per_page', 20))
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $this->page_meta['title'] = __('taxonomies::taxonomies.create_taxonomy');
        return template('admin/taxonomies/edit.twig', [
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(CreateTaxonomy $request)
    {
        $taxonomy = Taxonomy::query()
            ->create([
                'name' => $request->input('name'),
                'active' => $request->input('active', 0),
                'tree_structure' => $request->input('tree_structure', 0),
            ]);

        session()->flash('success', __('taxonomies::taxonomies.created'));
        return redirect(route('taxonomies.show', [$taxonomy->id]));
    }

    /**
     * Display the specified resource.
     *
     * @param Taxonomy $taxonomy
     * @return \Illuminate\Http\Response|TaxonomyList
     */
    public function show(Request $request, $taxonomy_slug)
    {
        $taxonomy = Taxonomy::find($taxonomy_slug);
        $taxonomy->load('children');

        if ($request->wantsJson()) {
            return new TaxonomyList($taxonomy);
        }
        $this->page_meta['title'] = __('taxonomies::taxonomies.view_taxonomy');
        return template('admin/taxonomies/show.twig', [
            'page_meta' => $this->page_meta,
            'taxonomy' => $taxonomy,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Taxonomy $taxonomy)
    {
        $this->page_meta['title'] = __('taxonomies::taxonomies.edit_taxonomy');
        return template('admin/taxonomies/edit.twig', [
            'page_meta' => $this->page_meta,
            'taxonomy' => $taxonomy,
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param Taxonomy $taxonomy
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateTaxonomy $request, Taxonomy $taxonomy)
    {
        $taxonomy->fill($request->input());
        $taxonomy->save();

        if ($request->wantsJson()) {
            $taxonomy->is_open = false;
            $taxonomy->edit = false;
            return $taxonomy;
        }
        session()->flash('success', __('taxonomies::taxonomies.updated'));
        return redirect(route('taxonomies.show', [$taxonomy->slug]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param Taxonomy $taxonomy
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, Taxonomy $taxonomy)
    {
        $taxonomy->delete();
        if ($request->wantsJson()) {
            return response('success');
        }
        session()->flash('success', __('taxonomies::taxonomies.removed'));
        return back();
    }

    /**
     * Add a child entry
     *
     * @param Request $request
     * @return TaxonomyList
     */
    public function addChild(Request $request)
    {
        /** @var Taxonomy $parent */
        $parent = Taxonomy::query()->find($request->input('parent_id'));
        $child = $parent->children()
            ->create([
                'name' => $request->input('name'),
                'description' => $request->input('description')
            ]);

        $parent->load('children');
        return new TaxonomyList($parent);
    }

    /**
     * List taxonomies based of filter
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function showList(Request $request)
    {
        $taxonomies = Taxonomy::query()->withDepth();

        if ($request->input('exclude', false)) {
            /** @var Taxonomy $exclude */
            $exclude = Taxonomy::query()->find($request->input('exclude', false));
            $root = $exclude->getRoot();

            $taxonomies->whereDescendantOf($root)
                ->where('id', '!=', $exclude->id)
                ->whereNotDescendantOf($exclude);
        }

        /*
         * Check if we are restricting the list by depth
         * Location selection only allows root level menu so it needs filtering by depth of 0
         */
        if ($request->input('of_depth', false)) {
            $taxonomies->having('depth', '=', $request->input('of_depth'));
        }

        $taxonomies = $taxonomies->orderBy('_lft')->get();

        /*
         * Reorder needs to have key/value map
         */
        if ($request->input('key_value_pair')) {
            $taxonomies = $taxonomies
                ->keyBy(function ($entry) {
                    return '_' . $entry->id;
                })
                ->map(function ($taxonomy) {
                    return str_repeat('-', $taxonomy->depth) . ' ' . $taxonomy->name;
                });
        }

        if ($request->input('prepend_root')) {
            if ($request->input('key_value_pair')) {
                $taxonomies->put(0, trans('taxonomies::taxonomies.root_level'));
            } else {
                $taxonomies->prepend([
                    'id' => '0',
                    'name' => trans('taxonomies::taxonomies.root_level'),
                ]);
            }
        }

        // Return a prepared response
        // Doing this instead of collection as collection is already sorted and we don't want the order to change
        return response()->json($taxonomies);
    }

    /**
     * Change the parent of a taxonomy
     *
     * @param Request $request
     */
    public function changeParent(Request $request)
    {
        /** @var Taxonomy $taxonomy */
        $taxonomy = Taxonomy::find($request->input('taxonomy_id'));

        if ($request->input('new_parent') == 0) {
            $taxonomy->getRoot()->appendNode($taxonomy);
        } else {
            /** @var Taxonomy $parent */
            $parent = Taxonomy::query()->find($request->input('new_parent'));
            $parent->appendNode($taxonomy);
        }
    }

    /**
     * Change the order of taxonomies
     *
     * @param Request $request
     */
    public function reorder(Request $request)
    {
        foreach ($request->input('taxonomies', []) as $key => $taxonomy_data) {
            /** @var Taxonomy $taxonomy */
            $taxonomy = Taxonomy::query()->find($taxonomy_data['id']);

            try {
                if (!empty($previous_taxonomy)) {
                    $taxonomy->afterNode($previous_taxonomy)->save();
                }
            } catch (\Exception $exception) {
                // Ignore impossible move and proceed
                // e.g. the order of the first child has not changed causes the exception
            }

            // set this taxonomy as previous so we can put next one besides this one
            $previous_taxonomy = $taxonomy;
        }
    }

    /**
     * Change the active state for taxonomy
     *
     * @param Taxonomy $taxonomy
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Illuminate\Http\Response
     */
    public function toggleActive(Taxonomy $taxonomy)
    {
        $taxonomy->active = !$taxonomy->active;
        $taxonomy->save();
        return response('success');
    }

}
