<?php

namespace Mtc\VehicleLookup\Drivers;

use Carbon\Carbon;
use Exception;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleLookup\Config\CapConfig;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\VehicleLookupResponse;

class CAP implements VehicleLookupDriver
{
    public function __construct(protected CapConfig $config)
    {
        //
    }

    public function lookup(string $registration_number, int $mileage): VehicleLookupResponse
    {
        return $this->parseResponse($registration_number, $this->runLookup($registration_number));
    }

    private function runLookup(string $registration_number): array
    {
        $result = $this->call('/DVLALookupVRM', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'vrm' => $registration_number,
        ], 'dvla');

        $dvla = $this->getXmlDataByElementName($result, 'DVLA');
        $data = $this->getXmlDataByElementName($result, 'CAP');

        // not found
        if (empty($data['VEHICLETYPE'])) {
            return [
                'cap_id' => null,
            ];
        }

        $result = [
            'type' => (string)$data['VEHICLETYPE'],
            'cap_id' => (string)$data['CAPID'],
            'make' => (string)$data['MANUFACTURER'],
            'model' => (string)$data['MODEL'],
            'derivative' => (string)$data['DERIVATIVE'],
            'fuel_type' => (string)$data['FUELTYPE'],
            'transmission' => (string)$data['TRANSMISSION'],
            'door_count' => (int)$data['DOORS'],
        ];

        $dvlaData = [];
        if (!empty($dvla['REGISTRATIONDATE'])) {
            $dvlaData = [
                'manufacture_year' => Carbon::createFromFormat('Ymd', (string)$dvla['REGISTRATIONDATE'])->format('Y'),
                'make' => (string)$dvla['MANUFACTURER'],
                'model' => (string)$dvla['MODEL'],
                'colour' => (string)$dvla['COLOUR'],
                'door_count' => (int)$dvla['DOORS'],
                'body_type' => (string)$dvla['BODYTYPE'],
            ];
        }

        foreach ($dvlaData as $key => $entry) {
            if (empty($result[$key]) && !empty($entry)) {
                $result[$key] = $entry;
            }
        }
        return $result;
    }

    protected function call(string $url, array $request_data, string $type = 'nvd')
    {
        $response = Http::asForm()->post($this->endpoint($url, $type), $request_data);

        if ($response->failed()) {
            Log::error('Failed to call CAP ' . $url, [
                'request_data' => $request_data,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ]);
            throw new Exception('Failed to perform call to ' . $url);
        }

        return $response->body();
    }

    private function endpoint(string $path, string $type = 'nvd'): string
    {
        if ($type === 'dvla') {
            return 'https://soap.cap.co.uk/DVLA/CAPDVLA.asmx/' . ltrim($path, '/');
        }
        return 'https://soap.cap.co.uk/Nvd/CapNvd.asmx/' . ltrim($path, '/');
    }


    private function parseResponse(string $registration_number, array $vehicle): VehicleLookupResponse
    {
        return new VehicleLookupResponse(
            $registration_number,
            make: $vehicle['make'] ?? null,
            model: $vehicle['model'] ?? null,
            derivative: $vehicle['derivative'] ?? null,
            manufacture_year: $vehicle['manufacture_year'] ?? null,
            colour: $vehicle['colour'] ?? null,
            body_type: $vehicle['body_type'] ?? null,
            fuel_type: $vehicle['fuel_type'] ?? null,
            transmission: $vehicle['transmission'] ?? null,
        );
    }

    /**
     * Extracts data from an XML string based on the element name.
     *
     * @param string $xml The XML string.
     * @param string $elementName The name of the element to extract data from.
     * @return mixed The extracted data or null if the element is not found.
     */
    public function getXmlDataByElementName(string $xml, string $elementName)
    {
        // Suppress errors and load the XML string
        $xmlObject = @simplexml_load_string($xml);

        // Check if XML was loaded successfully
        if ($xmlObject === false) {
            // Handle error, e.g., return null or throw an exception
            return null;
        }

        // Convert the SimpleXMLElement to JSON and then decode to an associative array
        $json = json_encode($xmlObject);
        $array = json_decode($json, true);

        // Search for the element by its name within the 'DATA' section
        if (isset($array['DATA'][$elementName])) {
            return $array['DATA'][$elementName];
        }

        // If element is not found, return null
        return null;
    }
}
