<?php

namespace Mtc\VehicleLookup\Drivers;

use App\Modules\Lookup\Contracts\VehicleLookupData;
use App\VehicleSpec\Contracts\VehicleOptionalEquipmentItem;
use App\VehicleSpec\Contracts\VehicleStandardEquipmentItem;
use App\VehicleSpec\Contracts\VehicleTechnicalDataItem;
use Carbon\Carbon;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Master\Models\VehicleMake as MasterMake;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\VehicleLookup\Config\CapConfig;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\VehicleLookupResponse;

class CAP implements VehicleLookupDriver
{
    public function __construct(protected CapConfig $config)
    {
        //
    }

    public function lookup(string $registration_number, int $mileage): VehicleLookupResponse
    {
        return $this->parseResponse($registration_number, $this->runLookup($registration_number));
    }

    private function runLookup(string $registration_number): array
    {
        $result = $this->call('/DVLALookupVRM', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'vrm' => $registration_number,
        ], 'dvla');

        $dvla = $this->getXmlDataByElementName($result, 'DVLA');
        $data = $this->getXmlDataByElementName($result, 'CAP');

        // not found
        if (empty($data['VEHICLETYPE'])) {
            return [
                'cap_id' => null,
            ];
        }

        $result = [
            'type' => (string)$data['VEHICLETYPE'],
            'cap_id' => (string)$data['CAPID'],
            'make' => (string)$data['MANUFACTURER'],
            'model' => (string)$data['MODEL'],
            'derivative' => (string)$data['DERIVATIVE'],
            'fuel_type' => (string)$data['FUELTYPE'],
            'transmission' => (string)$data['TRANSMISSION'],
            'door_count' => (int)$data['DOORS'],
        ];

        $dvlaData = [];
        if (!empty($dvla['REGISTRATIONDATE'])) {
            $dvlaData = [
                'manufacture_year' => Carbon::createFromFormat('Ymd', (string)$dvla['REGISTRATIONDATE'])->format('Y'),
                'make' => (string)$dvla['MANUFACTURER'],
                'model' => (string)$dvla['MODEL'],
                'colour' => (string)$dvla['COLOUR'],
                'door_count' => (int)$dvla['DOORS'],
                'body_type' => (string)$dvla['BODYTYPE'],
            ];
        }

        foreach ($dvlaData as $key => $entry) {
            if (empty($result[$key]) && !empty($entry)) {
                $result[$key] = $entry;
            }
        }
        return $result;
    }

    protected function call(string $url, array $request_data, string $type = 'nvd')
    {
        $response = Http::asForm()->post($this->endpoint($url, $type), $request_data);

        if ($response->failed()) {
            Log::error('Failed to call CAP ' . $url, [
                'request_data' => $request_data,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ]);
            throw new Exception('Failed to perform call to ' . $url);
        }

        return $response->body();
    }

    private function endpoint(string $path, string $type = 'nvd'): string
    {
        if ($type === 'dvla') {
            return 'https://soap.cap.co.uk/DVLA/CAPDVLA.asmx/' . ltrim($path, '/');
        }

        if ($type === 'vehicles') {
            return 'https://soap.cap.co.uk/Vehicles/CapVehicles.asmx/' . ltrim($path, '/');
        }

        return 'https://soap.cap.co.uk/Nvd/CapNvd.asmx/' . ltrim($path, '/');
    }


    private function parseResponse(string $registration_number, array $vehicle): VehicleLookupResponse
    {
        return new VehicleLookupResponse(
            $registration_number,
            make: $vehicle['make'] ?? null,
            model: $vehicle['model'] ?? null,
            derivative: $vehicle['derivative'] ?? null,
            manufacture_year: $vehicle['manufacture_year'] ?? null,
            colour: $vehicle['colour'] ?? null,
            body_type: $vehicle['body_type'] ?? null,
            fuel_type: $vehicle['fuel_type'] ?? null,
            transmission: $vehicle['transmission'] ?? null,
        );
    }

    /**
     * Extracts data from an XML string based on the element name.
     *
     * @param string $xml The XML string.
     * @param string $elementName The name of the element to extract data from.
     * @return mixed The extracted data or null if the element is not found.
     */
    public function getXmlDataByElementName(string $xml, string $elementName)
    {
        // Suppress errors and load the XML string
        $xmlObject = @simplexml_load_string($xml);

        // Check if XML was loaded successfully
        if ($xmlObject === false) {
            // Handle error, e.g., return null or throw an exception
            return null;
        }

        $result = $xmlObject->xpath("//{$elementName}");

        return $result ? json_decode(json_encode($result), true) : null;
    }

    public function getMakes($current = 'false')
    {
        if (empty($this->config->subscriberId())) {
            return [];
        }

        $result = $this->call('/GetCapMan', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => 'cars',
            'justCurrentManufacturers' => $current,
            'bodyStyleFilter' => ''
        ], 'vehicles');

        $manufacturers = $this->getXmlDataByElementName($result, 'Table');

        $manufacturerData = [];
        if ($manufacturers) {
            foreach ($manufacturers as $manufacturer) {
                $name = ucwords(strtolower(trim((string)$manufacturer['CMan_Name'])), " -");
                $manufacturerData[$name] = [
                    'id' => (string)$manufacturer['CMan_Code'],
                    'name' => $name,
                ];
            }
        }

        $lcvResult = $this->call('/GetCapMan', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => 'lcv',
            'justCurrentManufacturers' => $current,
            'bodyStyleFilter' => ''
        ], 'vehicles');

        $lcvManufacturers = $this->getXmlDataByElementName($lcvResult, 'Table');

        if ($lcvManufacturers) {
            foreach ($lcvManufacturers as $manufacturer) {
                $name = ucwords(strtolower(trim((string)$manufacturer['CMan_Name'])), " -");
                $manufacturerData[$name]['lcv_id'] = (string)$manufacturer['CMan_Code'];
                $manufacturerData[$name]['name'] = $name;
            }
        }

        return $manufacturerData;
    }

    public function getModels(MasterMake $make, $current = 'false')
    {
        if (empty($this->config->subscriberId())) {
            return [];
        }

        $modelsData = [];
        if (!empty($make->cap_id)) {
            $result = $this->call('/GetCapMod', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => 'cars',
                'manRanCode' => (int)$make->cap_id,
                'manRanCodeIsMan' => 'true',
                'justCurrentModels' => $current,
                'bodyStyleFilter' => '',
            ], 'vehicles');

            $models = $this->getXmlDataByElementName($result, 'Table');

            if ($models) {
                foreach ($models as $model) {
                    $name = ucwords(strtolower(trim((string)$model['CMod_Name'])), " -");
                    $modelsData[$name]['id'] = (string)$model['CMod_Code'];
                    $modelsData[$name]['name'] = $name;
                }
            }
        }

        if (!empty($make->lcv_cap_id)) {
            $result = $this->call('/GetCapMod', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => 'lcv',
                'manRanCode' => (int)$make->lcv_cap_id,
                'manRanCodeIsMan' => 'true',
                'justCurrentModels' => $current,
                'bodyStyleFilter' => '',
            ], 'vehicles');

            $models = $this->getXmlDataByElementName($result, 'Table');

            if ($models) {
                foreach ($models as $model) {
                    $name = ucwords(strtolower(trim((string)$model['CMod_Name'])), " -");
                    $modelsData[$name]['id'] = (string)$model['CMod_Code'];
                    $modelsData[$name]['name'] = $name;
                }
            }
        }

        return $modelsData;
    }

    public function findDerivatives(VehicleMake $make, VehicleModel $model): array
    {
        if (empty($this->config->subscriberId())) {
            return [];
        }

        $result = $this->call('/GetCapDer', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => 'cars',
            'modCode' => $model->cap_id,
            'justCurrentDerivatives' => 'false',
        ], 'vehicles');

        $derivativeData = [];
        $derivatives = $this->getXmlDataByElementName($result, 'Table');
        if (!empty($derivatives)) {
            foreach ($derivatives as $derivative) {
                $derivativeData[] = [
                    'id' => (string)$derivative['CDer_ID'],
                    'name' => trim($derivative['CDer_Name'] . ' ' . $derivative['ModelYearRef'])
                ];
            }
        }

        return $derivativeData;
    }

    public function findVariants(VehicleMake $make, VehicleModel $model, string $derivative = ''): array
    {
        if (empty($this->config->subscriberId())) {
            return [];
        }

        $result = $this->call('/GetCapDer', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => 'cars',
            'modCode' => $model->cap_id,
            'justCurrentDerivatives' => 'false',
        ], 'vehicles');

        $derivativeData = [];
        $derivatives = $this->getXmlDataByElementName($result, 'Table');
        if (!empty($derivatives)) {
            foreach ($derivatives as $derivative) {
                $derivativeData[] = [
                    'id' => (string)$derivative['CDer_ID'],
                    'name' => trim($derivative['CDer_Name'] . ' ' . $derivative['ModelYearRef'])
                ];
            }
        }

        return $derivativeData;
    }

    public function getTechnicalData(string $variant): VehicleLookupData
    {
        if (empty($this->config->subscriberId())) {
            return new VehicleLookupData();
        }

        $data = $this->getBulkTechnicalData($variant);

        return new VehicleLookupData(
            make: $data['make'] ?? null,
            model: $data['model'] ?? null,
            derivative: $data['derivative'] ?? null,
            body_type: $data['body_style'] ?? null,
            fuel_type: $data['fuel_type'] ?? null,
            transmission: $data['transmission'] ?? null,
            doors: $data['doors'] ?? null,
            standard_equipment: $this->getStandardEquipment($variant),
            optional_equipment: $this->getOptionsBundle($variant),
            technical_data: $this->technicalData($variant),
        );
    }

    protected function technicalData($id, $database = 'cars'): Collection
    {
        $technical = [];

        try {
            $result = $this->call('/GetTechnicalData', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => $database,
                'capid' => $id,
                'techDate' => date('d/M/Y'),
                'justCurrent' => 'false'
            ]);
            $data = $this->getXmlDataByElementName($result, 'Tech');

            if (!empty($data)) {
                foreach ($data as $td) {
                    if (!empty($td['Dc_Description']) && !empty($td['tech_value_string'])) {
                        $technical[] = new VehicleTechnicalDataItem(
                            (string)$td['Tech_TechCode'],
                            (string)$td['Dc_Description'],
                            (string)$td['DT_LongDescription'],
                            (string)$td['tech_value_string']
                        );
                    }
                }
            }
        } catch (Exception $exception) {
            Log::debug('CAP lookup failed: ' . $exception->getMessage());
        }

        return collect($technical)->unique('description');
    }

    protected function getDescriptionFromID($id, $database = 'cars'): array
    {
        $result = $this->call('/GetCapDescriptionFromId', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => $database,
            'capid' => $id,
            'techDate' => date('d/M/Y'),
            'justCurrent' => 'false'
        ], 'vehicles');

        $description = [];
        $data = $this->getXmlDataByElementName($result, 'Table')[0];

        $description['manufacturer'] = (string)$data['CVehicle_ManText'];
        $description['short_model'] = (string)$data['CVehicle_ShortModText'];
        $description['model'] = (string)$data['CVehicle_ModText'];
        $description['derivative'] = (string)$data['CVehicle_DerText'];
        $description['derivative_name'] = (string)$data['CVehicle_ShortDerText'];
        $description['man_text_code'] = (int)$data['CVehicle_ManTextCode'];
        $description['short_mod_id'] = (int)$data['CVehicle_ShortModID'];
        $description['mod_text_code'] = (int)$data['CVehicle_ModTextCode'];

        return $description;
    }

    protected function getP11DData($id, $database = 'cars')
    {
        $result = $this->call('/GetP11DData', [
            'subscriberId' => $this->config->subscriberId(),
            'password' => $this->config->password(),
            'database' => $database,
            'capid' => $id,
            'year1' => date("Y"),
            'year2' => date('Y', strtotime('+1 year')),
            'year3' => date('Y', strtotime('+2 years')),
        ]);

        $p11Data = [];
        $data = $this->getXmlDataByElementName($result, 'P11D');

        foreach ($data[0] as $t) {
            foreach ($t as $k => $v) {
                $p11Data[$k] = (string)$v;
            }
        }

        return $data;
    }

    protected function getBulkTechnicalData($id, $database = 'cars'): array
    {
        $vehicle = [];

        try {
            $result = $this->call('/GetBulkTechnicalData', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => $database,
                'capidList' => $id,
                'specDateList' => date('d/M/Y'),
                'techDataList' => '',
                'returnVehicleDescription' => 'true',
                'returnCaPcodeTechnicalItems' => 'true',
                'returnCostNew' => 'false',
            ]);

            $techTable = $this->getXmlDataByElementName($result, 'Tech_Table');
            if (!empty($techTable)) {
                $data = $techTable[0];
                $vehicle['make'] = (string)$data['Manufacturer'];
                $vehicle['model'] = (string)$data['Model'];
                $vehicle['derivative'] = (string)$data['Derivative'];
                $vehicle['body_style'] = (string)$data['BodyStyle'];
                $vehicle['fuel_type'] = (string)$data['FuelType'];
                $vehicle['transmission'] = (string)$data['Transmission'];
                $vehicle['doors'] = (int)$data['Doors'];
            }
        } catch (Exception $exception) {
            Log::debug('CAP lookup failed: ' . $exception->getMessage());
        }

        return $vehicle;
    }

    protected function getStandardEquipment($id, $database = 'cars'): Collection
    {
        $standard = [];

        try {
            $result = $this->call('/GetStandardEquipment', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => $database,
                'capid' => $id,
                'seDate' => date('d/M/Y'),
                'justCurrent' => 'false',
            ]);

            $data = $this->getXmlDataByElementName($result, 'SE');

            if (!empty($data)) {
                foreach ($data as $se) {
                    if (null != $se['Dc_Description']
                        && null != $se['Do_Description']
                    ) {
                        $standard[] = new VehicleStandardEquipmentItem(
                            (string)$se['Se_OptionCode'],
                            substr((string)$se['Do_Description'], 0, 500),
                            (string)$se['Dc_Description']
                        );
                    }
                }
            }
        } catch (Exception $exception) {
            Log::debug('CAP lookup failed: ' . $exception->getMessage());
        }

        return collect($standard)->unique('description');
    }

    protected function getOptionsBundle($id, $database = 'cars'): Collection
    {
        $optional = [];

        try {
            $result = $this->call('/GetCapOptionsBundle', [
                'subscriberId' => $this->config->subscriberId(),
                'password' => $this->config->password(),
                'database' => $database,
                'capid' => $id,
                'optionDate' => date('d/M/Y'),
                'justCurrent' => 'false',
                'descriptionRs' => 'true',
                'optionsRs' => 'true',
                'relationshipsRs' => 'true',
                'packRs' => 'true',
                'technicalRs' => 'false',
            ]);

            $data = $this->getXmlDataByElementName($result, 'NVDBundle')[0];

            foreach ($data['Options'] as $option) {
                $optional[] = new VehicleOptionalEquipmentItem(
                    (string)$option['Opt_OptionCode'],
                    substr((string)$option['Do_Description'], 0, 500),
                    (string)$option['Dc_Description'],
                    $option['Opt_Basic'] ?? 0,
                    (string)$option['Opt_VAT'] ?? 0
                );
            }
        } catch (Exception $exception) {
            Log::debug('CAP lookup failed: ' . $exception->getMessage());
        }

        return collect($optional)->unique('description');
    }
}
