<?php

namespace Mtc\VehicleLookup\Drivers;

use App\Modules\Lookup\Contracts\VehicleLookupData;
use App\VehicleSpec\Contracts\VehicleStandardEquipmentItem;
use App\VehicleSpec\Contracts\VehicleTechnicalDataItem;
use App\VehicleSpec\Services\MotorCheckSpecProvider;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\AutomotiveSpecs\VehicleSpecification;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\MotorCheck\Contracts\MotorSpecsToVehicleSpecificationContract;
use Mtc\MotorCheck\Services\MotorSpecsApi;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\VehicleLookupResponse;

class MotorCheck extends MotorSpecsApi implements VehicleLookupDriver
{
    protected array $technicalEquipmentTypes = [
        'Technical',
        'Dimensions',
    ];

    protected array $standardEquipmentTypes = [
        'Interior',
        'Exterior',
        'Safety',
        'Security',
        'Others',
    ];

    public function __construct(protected array $config = [])
    {

    }

    public function lookup(string $registration_number, int $mileage): VehicleLookupResponse
    {
        $specs = $this->getSpecs($registration_number, $mileage);
        $vehicleData = App::make(MotorSpecsToVehicleSpecificationContract::class)->map($specs);

        return $this->parseResponse($registration_number, $vehicleData);
    }

    private function parseResponse(string $registration_number, VehicleSpecification $vehicle): VehicleLookupResponse
    {
        $motorSpecProvider = App::make(MotorCheckSpecProvider::class);

        return new VehicleLookupResponse(
            $registration_number,
            mileage: $vehicle->odometer_value,
            make: $vehicle->make,
            model: $vehicle->model,
            derivative: $vehicle->derivative,
            colour: $vehicle->colour,
            fuel_type: $vehicle->fuel_type,
            transmission: $vehicle->transmission,
            model_year: $vehicle->model_year,
            registration_date: $vehicle->registration_date,
            mpg: $vehicle->mpg,
            co2: $vehicle->co2,
            engine_capacity_cc: $vehicle->engine_size,
            body_style: $vehicle->body_style,
            drivetrain: $vehicle->drivetrain,
            standard_equipment: $motorSpecProvider->processStandardEquipment($vehicle->feature_list),
            technical_data: $motorSpecProvider->processTechnicalData($vehicle->feature_list)
        );
    }

    public function findDerivatives(VehicleMake $make, VehicleModel $model): array
    {
        return [];
    }

    public function findVariants(VehicleMake $make, VehicleModel $model, string $derivative = ''): array
    {
        if (empty($this->config['username'])) {
            return [];
        }

        $variants = [];

        $json = json_encode([
            "country" => $this->config['country'] ?? null,
            'make' => $make->name,
            'model' => $model->name,
            "fuel" => "",
            "trim" => "",
            "engineSize" => "",
            "doors" => "",
            "body" => "",
            "display" => "c",
            "query" => 1,
            "maxResult" => 100
        ], JSON_THROW_ON_ERROR);

        try {
            $response = Http::withToken($this->getAccessToken())
                ->withHeaders($this->headers('vnd.make-model-select.v2+json'))
                ->withBody($json, 'application/vnd.make-model-select.v2+json')
                ->post($this->endpoint('make-model-select/query'))
                ->throw()
                ->json();

            if(!empty($response['results'])){
                foreach($response['results'] as $result){
                    $variants[] = [
                        'id' => $result['id'],
                        'name' => $result['version']
                    ];
                    Cache::put('motor-check-' . $result['id'], $result, now()->addMinutes(60));
                }
            }

            usort($variants, function($a, $b) {
                return strcmp($a['name'], $b['name']);
            });
        } catch (Exception $exception) {
            Log::debug($exception->getMessage());
        }

        return $variants;
    }

    public function getTechnicalData(string $variant): VehicleLookupData
    {
        if (empty($this->config['username'])) {
            return new VehicleLookupData();
        }

        if(!Cache::has('motor-check-' . $variant)){
            return new VehicleLookupData();
        }

        $data = Cache::get('motor-check-' . $variant);

        $features = collect([]);
        $standardEquipment = collect([]);
        $technicalData = collect([]);
        try {
            $standardSpecs = Http::withToken($this->getAccessToken())
                ->withHeaders($this->headers('vnd.specs.v2+json'))
                ->withBody(json_encode([
                    "country" => $this->config['country'] ?? null,
                    'vehicleId' => $variant
                ], JSON_THROW_ON_ERROR), 'application/vnd.specs.v2+json')
                ->post($this->endpoint('specs/standard-by-id'))
                ->throw()
                ->json();

            if(!empty($standardSpecs['topFeatures'])){
                $features = collect(array_unique(array_column($standardSpecs['topFeatures'], 'name')));
            }

            if(!empty($standardSpecs['standardSpecification'])){
                $standardEquipment = $this->processStandardEquipment($standardSpecs['standardSpecification']);
                $technicalData = $this->processTechnicalData($standardSpecs['standardSpecification']);
            }
        } catch (Exception $exception) {
            Log::debug($exception->getMessage());
        }

        return new VehicleLookupData(
            make: $data['make'],
            model: $data['model'],
            generation: $data['modelGeneration'],
            derivative: $data['version'],
            trim: $data['trimLevel'],
            body_type: $data['bodyType'],
            fuel_type: $data['fuel'],
            transmission: $data['transmission'],
            seats: $data['seatingCapacity'],
            doors: $data['numberDoors'],
            drive_type: $data['drivenWheels'],
            engine_capacity_cc: $data['engineCC'],
            features: $features,
            standard_equipment: $standardEquipment,
            technical_data: $technicalData
        );
    }

    /**
     * Map standard equipment features
     *
     * @param array $features
     * @return Collection
     */
    protected function processStandardEquipment(array $features = []): Collection
    {
        $standardEquipment = [];

        foreach($features as $category){
            foreach($category as $categoryName => $categoryFeatures){
                if(in_array($categoryName, $this->standardEquipmentTypes)){
                    foreach($categoryFeatures as $feature){
                        $standardEquipment[] = new VehicleStandardEquipmentItem(
                            $feature['id'],
                            substr($feature['description'], 0, 500),
                            $categoryName
                        );
                    }
                }
            }
        }

        return collect($standardEquipment)->unique('description');
    }

    /**
     * Process technical data response
     *
     * @param array $features
     * @return Collection
     */
    protected function processTechnicalData(array $features): Collection
    {
        $technicalData = [];

        foreach($features as $category){
            foreach($category as $categoryName => $categoryFeatures){
                if(in_array($categoryName, $this->technicalEquipmentTypes)){
                    foreach($categoryFeatures as $feature){
                        $technicalData[] = new VehicleTechnicalDataItem(
                            $feature['name'],
                            $categoryName,
                            $feature['name'],
                            substr($feature['description'], 0, 500)
                        );
                    }
                }
            }
        }

        return collect($technicalData)->unique('description');
    }
}
