<?php

namespace Mtc\VehicleLookup;

use App\Facades\Settings;
use Illuminate\Support\Facades\App;
use Mtc\MercuryDataModels\VehicleMake;
use Mtc\MercuryDataModels\VehicleModel;
use Mtc\VehicleLookup\Contracts\VehicleLookupDriver;
use Mtc\VehicleLookup\Drivers\AutoTrader;
use Mtc\VehicleLookup\Drivers\BeepBeep;
use Mtc\VehicleLookup\Drivers\CAP;
use Mtc\VehicleLookup\Drivers\BCA;
use Mtc\VehicleLookup\Drivers\EVA;
use Mtc\VehicleLookup\Drivers\MotorCheck;
use Mtc\VehicleLookup\Drivers\NullDriver;
use Mtc\VehicleLookup\Drivers\UKMOT;

class VehicleLookupService
{
    private VehicleLookupDriver $active_driver;

    public function findByVRM(string $registration_number, int $mileage): VehicleLookupResponse
    {
        return $this->getDriver()->lookup($registration_number, $mileage);
    }

    public function findDerivatives(VehicleMake $make, VehicleModel $model): array
    {
        return $this->getDriver()->findDerivatives($make, $model);
    }

    public function findVariants(VehicleMake $make, VehicleModel $model, $derivative = ''): array
    {
        return $this->getDriver()->findVariants($make, $model, $derivative);
    }

    public function getTechnicalData($variant)
    {
        return $this->getDriver()->getTechnicalData($variant);
    }

    public function hasActiveDriver(): bool
    {
        return !empty($this->findActiveDriver());
    }

    private function getDriver(): VehicleLookupDriver
    {
        if (!isset($this->active_driver)) {
            $this->loadDriver();
        }

        return $this->active_driver;
    }

    private function loadDriver(): void
    {
        $this->active_driver = match ($this->findActiveDriver()) {
            'auto-trader' => new AutoTrader(App::make(config('lookup.drivers')['auto-trader']['config'])),
            'bca' => new BCA(App::make(config('lookup.drivers')['bca']['config'])),
            'cap' => new CAP(App::make(config('lookup.drivers')['cap']['config'])),
            'beep-beep' => new BeepBeep(App::make(config('lookup.drivers')['beep-beep']['config'])),
            'uk-mot' => new UKMOT(),
            'motor-check' => new MotorCheck(
                App::make(config('lookup.drivers')['motor-check']['config'])->toArray()
            ),
            'eva' => new EVA(App::make(config('lookup.drivers')['eva']['config'])),
            default => new NullDriver(),
        };
    }

    public function findActiveDriver(): ?string
    {
        return collect([
            'auto-trader' => 'lookup-auto-trader-enabled',
            'cap' => 'lookup-cap-enabled',
            'bca' => 'lookup-bca-enabled',
            'uk-mot' => 'lookup-uk-mot',
            'beep-beep' => 'lookup-beepbeep-enabled',
            'motor-check' => 'lookup-motor-check-enabled',
            'eva' => 'lookup-eva-enabled',
        ])->filter(fn($setting) => Settings::get($setting))
            ->keys()
            ->first();
    }
}
