<?php

namespace Mtc\VehicleReservations;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\InvoiceFactory;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\VehicleReservations\Mail\NewReservationCustomerMail;
use Mtc\VehicleReservations\Mail\NewReservationTeamMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledCustomerMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledTeamMail;

class ReservationRepository
{
    public function getVehicleAction(Vehicle $vehicle): array
    {
        return [
            'type' => 'link',
            'vehicle' => $vehicle->slug,
        ];
    }

    public function create(array $data): Model
    {
        $data['status'] = 'pending';
        $data['reference'] = strtoupper(Str::random(12));
        return Reservation::query()->create($data);
    }

    public function createInvoice(Reservation $reservation): InvoiceRepositoryContract
    {
        return (new InvoiceFactory())->create($reservation);
    }

    public function updateDetails(Reservation $reservation, array $details)
    {
        $reservation->update($details);
        return $reservation;
    }

    public function markConfirmed(Reservation $reservation, float $amount_paid): bool
    {
        try {
            $reservation->update([
                'status' => 'confirmed',
                'confirmed_at' => Carbon::now(),
                'amount' => $amount_paid,
            ]);

            $reservation->vehicle->update(['is_reserved' => true]);

            $this->notifyCustomer($reservation);
            $this->notifyAdminTeam($reservation);

            return true;
        } catch (\Exception $exception) {
            Log::error('Unable to mark reservation as confirmed', [
                $reservation,
                $exception->getMessage(),
                $exception->getTrace(),
            ]);
            return false;
        }
    }

    public function cancel(Reservation $reservation): bool
    {
        if ($this->shouldRefund($reservation) && $this->canRefund($reservation)) {
            return $this->refund($reservation);
        }

        $reservation->update(['status' => 'cancelled']);
        if ($this->vehicleHasNoReservation($reservation->vehicle)) {
            $reservation->vehicle->update(['is_reserved' => false]);
        }

        return false;
    }

    private function notifyCustomer(Reservation $reservation): void
    {
        $mailable = match ($reservation->status) {
            'confirmed' => new NewReservationCustomerMail($reservation),
            'cancelled' => new ReservationCancelledCustomerMail($reservation),
        };
        Mail::to($reservation->email)->send($mailable);
    }

    private function notifyAdminTeam(Reservation $reservation): void
    {
        $mailable = match ($reservation->status) {
            'confirmed' => new NewReservationTeamMail($reservation),
            'cancelled' => new ReservationCancelledTeamMail($reservation),
        };
        $team_emails = $this->getTeamEmails();
        if (!empty($team_emails)) {
            Mail::to($team_emails)->send($mailable);
        }
    }

    private function shouldRefund(Reservation $reservation): bool
    {
        return $reservation->status == 'confirmed';
    }

    private function canRefund(Reservation $reservation): bool
    {
        return $reservation->invoice->isRefundable();
    }

    private function refund(Reservation $reservation): bool
    {
        return $invoice_repository->refund($reservation->invoice);
    }

    private function vehicleHasNoReservation(Vehicle $vehicle): bool
    {
        return $vehicle->reservations()
            ->whereNotIn('status', ['pending', 'cancelled'])
            ->exists();
    }

    private function getTeamEmails(): array
    {
        return [];
    }

}
