<?php

namespace Mtc\VehicleReservations;

use App\Facades\Settings;
use App\IntegrationRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\InvoiceFactory;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\VehicleReservations\Mail\NewReservationCustomerMail;
use Mtc\VehicleReservations\Mail\NewReservationTeamMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledCustomerMail;
use Mtc\VehicleReservations\Mail\ReservationCancelledTeamMail;
use Mtc\Vyne\Vyne;

class ReservationRepository
{
    public function getVehicleAction(Vehicle $vehicle): array
    {
        return [
            'type' => 'link',
            'vehicle' => $vehicle->slug,
        ];
    }

    public function create(array $data): Model
    {
        $data['status'] = 'pending';
        $data['reference'] = strtoupper(Str::random(12));
        return Reservation::query()->create($data);
    }

    public function createInvoice(Reservation $reservation): InvoiceRepositoryContract
    {
        return (new InvoiceFactory())->create($reservation);
    }

    public function updateDetails(Reservation $reservation, array $details): Reservation
    {
        $reservation->fill($details);
        $this->updateInvoice($reservation);
        $reservation->save();
        return $reservation;
    }

    public function addEmailWhenRequired(Reservation $reservation, ?string $email = null): void
    {
        if (empty($reservation->email) && $email) {
            $reservation->update([
                'email' => $email
            ]);
        }
    }

    public function markConfirmed(Reservation $reservation, float $amount_paid): bool
    {
        try {
            $reservation->update([
                'status' => 'confirmed',
                'confirmed_at' => Carbon::now(),
                'amount' => $amount_paid,
            ]);

            if ($this->shouldMarkVehicleReservedOnConfirmation()) {
                $reservation->vehicle->update(['is_reserved' => true]);
            }
            return true;
        } catch (\Exception $exception) {
            Log::error('Unable to mark reservation as confirmed', [
                $reservation,
                $exception->getMessage(),
                $exception->getTrace(),
            ]);
            return false;
        }
    }

    public function cancel(Reservation $reservation): bool
    {
        if ($this->shouldRefund($reservation) && $this->canRefund($reservation)) {
            return $this->refund($reservation);
        }

        $reservation->update(['status' => 'cancelled']);
        if ($this->vehicleHasNoReservation($reservation->vehicle)) {
            $reservation->vehicle->update(['is_reserved' => false]);
        }

        return false;
    }

    public function getPaymentConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        return match ($this->getPaymentProvider()) {
            'stripe' => $this->stripeConfig($reservation, $invoice_repository),
            'vyne' => $this->vyneConfig($reservation, $invoice_repository),
            default => [],
        };
    }

    protected function getPaymentProvider(): string
    {
        // todo: dynamic
        return 'stripe';
    }

    protected function stripeConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        return [
            'provider' => 'stripe',
            'invoice_id' => $invoice_repository->getId(),
            'amount' => $invoice_repository->getOutstandingAmount() * 100,
            'country' => Settings::get('app-details-country'),
            'currency' => $invoice_repository->getCurrency(),
            'item_name' => 'Reservation fee for ' . $invoice_repository->getModel()->items->pluck('name')->implode(','),
            'stripe_public_key' => App::make(config('stripe.config'))->publicKey(),
            'customer' => null,
        ];
    }


    protected function vyneConfig(Reservation $reservation, InvoiceRepositoryContract $invoice_repository): array
    {
        /** @var Vyne $vyne */
        $vyne = App::make(Vyne::class);
        return [
            'provider' => 'vyne',
            'payment_details' => $vyne->initializePayment($invoice_repository),
            'invoice_id' => $invoice_repository->getId(),
            'amount' => $invoice_repository->getOutstandingAmount() * 100,
            'country' => Settings::get('app-details-country'),
            'currency' => $invoice_repository->getCurrency(),
            'item_name' => 'Reservation fee for ' . $invoice_repository->getModel()->items->pluck('name')->implode(','),
            'merchant_id' => Settings::get('sales-vyne-merchant-id'),
            'customer' => null,
        ];
    }

    private function shouldRefund(Reservation $reservation): bool
    {
        return $reservation->status == 'confirmed';
    }

    private function canRefund(Reservation $reservation): bool
    {
        return $reservation->invoice->isRefundable();
    }

    private function refund(Reservation $reservation): bool
    {
        return $invoice_repository->refund($reservation->invoice);
    }

    private function vehicleHasNoReservation(Vehicle $vehicle): bool
    {
        return $vehicle->reservations()
            ->whereNotIn('status', ['pending', 'cancelled'])
            ->exists();
    }


    private function shouldMarkVehicleReservedOnConfirmation(): bool
    {
        return Settings::get('mark-vehicle-reserved-on-confirmation', true);
    }

    private function updateInvoice(Reservation $reservation): void
    {
        if ($reservation->isDirty('email')) {
            $reservation->invoice->update([
                'email' => $reservation->email
            ]);
        }
    }
}
