<?php

namespace Mtc\VehicleReservations;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Collection;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Invoice;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\VehicleReservations\Factories\ReservationFactory;

/**
 * @property string $status
 * @property string $email
 * @property string $contact_number
 * @property ?Vehicle $vehicle
 * @property ?Invoice $invoice
 */
class Reservation extends Model implements PayableContract
{
    use HasFactory;

    /**
     * Mass assignable variables
     *
     * @var array
     */
    protected $fillable = [
        'vehicle_id',
        'customer_id',
        'reference',
        'status',
        'amount',
        'amount_ex_vat',
        'amount_in_currency',
        'currency',
        'email',
        'name',
        'contact_number',
        'vat_value',
        'data',
        'confirmed_at',
        'confirmation_sent_at',
        'ingested_at',
        'attempts',
    ];

    /**
     * Cast variables to specific types
     *
     * @var array
     */
    protected $casts = [
        'confirmed_at' => 'datetime',
        'data' => 'array',
    ];

    protected static function newFactory(): ReservationFactory
    {
        return ReservationFactory::new();
    }

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function invoice(): MorphOne
    {
        return $this->morphOne(Invoice::class, 'payable');
    }

    public function getPayableClass(): string
    {
        return self::getMorphClass();
    }

    public function getReference(): string
    {
        return $this->reference;
    }

    public function getMember()
    {
        return $this->customer_id;
    }

    public function getTemplate(): string
    {
        return '';
    }

    public function getAmount(): float
    {
        return Settings::get('sales-reserve_price', 99);
    }

    public function getItems(): Collection
    {
        return collect([
            [
                'purchasable_id' => $this->vehicle_id,
                'purchasable_type' => 'vehicle',
                'name' => $this->vehicle->vrm_condensed . ' - ' . $this->vehicle->title,
                'unit_price' => Settings::get('sales-reserve_price', 99),
                'paid_price' => Settings::get('sales-reserve_price', 99),
                'quantity' => 1,
                'vat_rate' => 0.2,
                'attribute_fields' => [],
            ],
        ]);
    }

    public function getLines(): array
    {
        return [];
    }

    public function getAddress(): array
    {
        $name = explode(' ', $this->name);
        $first_name = array_shift($name);
        return [
            'first_name' => $first_name,
            'last_name' => implode(' ', $name),
        ];
    }

    public function getAdditionalDetails()
    {
        return [];
    }

    public function useExVat(): bool
    {
        return false;
    }
}
