<?php

namespace Mtc\VehicleSavings;

use App\Facades\Settings;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Vehicle;
use Mtc\VehicleSavings\Contracts\SavingsCalculator;

class SavingsCalculatorDefault implements SavingsCalculator
{
    protected $savings_floor;
    protected $savings_ceiling;

    public function __construct() {

        $this->savings_floor = $this->getSavingsFloor();
        $this->savings_ceiling = $this->getSavingsCeiling();

        if ($this->savings_floor <= 0) {
            $this->savings_floor = 0;
        }

        if ($this->savings_ceiling <= 0) {
            $this->savings_ceiling = PHP_INT_MAX;
        }
    }

    /**
     * @param Vehicle $vehicle
     * @return Collection
     */
    public function getSavings(Vehicle $vehicle): Collection
    {
        $savings = collect([]);

        if (!$this->enabled()
            || !$this->VehicleIsEligibleForSavings($vehicle)) {
            return $savings;
        }

        return $this->calculateSavings($vehicle);
    }

    /**
     * @param Vehicle $vehicle
     * @return bool
     */
    public function VehicleIsEligibleForSavings(Vehicle $vehicle): bool
    {
        return true;
    }

    /**
     * @param Vehicle $vehicle
     * @return Collection
     */
    protected function calculateSavings(Vehicle $vehicle): Collection
    {
        $total = 0;
        $savings = collect([
            'saving_original_price' => null,
            'saving_rrp' => null,
        ]);

        $saving_original_price = $vehicle->original_price - $vehicle->price;
        $saving_rrp = $vehicle->rrp_price - $vehicle->price;

        // prefer saving from original price, falling back to saving from RRP
        if ($saving_original_price > 0) {
            $saving_rrp = null;
        } elseif ($saving_rrp > 0) {
            $saving_original_price = null;
        }

        if (Settings::get('vehicle-savings-enabled-previous-price')) {
            $savings->put('saving_original_price', $saving_original_price);
        }

        if (Settings::get('vehicle-savings-enabled-rrp')) {
            $savings->put('saving_rrp', $saving_rrp);
        }

        // apply floors and ceilings to savings
        $savings_sanitised = collect($savings)->map(function ($saving) use (&$total) {

            // don't process null savings items
            if ($saving == null) {
                return null;
            }

            if ($saving < $this->savings_floor) {
                $saving = $this->savings_floor;
            }

            if ($saving > $this->savings_ceiling) {
                $saving = $this->savings_ceiling;
            }

            $total += $saving;

            return $saving;
        });

        $savings_sanitised->put('saving_total', $total);

        return $savings_sanitised;
    }

    /**
     * @return bool
     */
    protected function enabled(): bool
    {
        return Settings::get('vehicle-savings-enabled', false) === true;
    }

    /**
     * @return int
     */
    protected function getSavingsFloor(): int
    {
        return Settings::get('vehicle-savings-floor', 0);
    }

    /**
     * @return int
     */
    protected function getSavingsCeiling(): int
    {
        return Settings::get('vehicle-savings-ceiling', PHP_INT_MAX);
    }
}
