<?php

namespace Mtc\VehicleValuation;

use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\AutoTraderConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;

class AutoTrader implements Contracts\ValuationDriver
{
    public function __construct(protected AutoTraderConfig $config)
    {
        //
    }

    public function getValuation(string $registration, int $mileage): ValuationResponse
    {
        $valuation = Http::withHeaders($this->headers())
            ->get($this->endpoint($registration, $mileage))
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('CapData Valuation Request failed', [
                    'message' => $response->body(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);
                throw new ValuationRequestFailed('AutoTrader Valuation request failed');
            })->json();

        return new ValuationResponse(
            provider: 'auto-trader',
            registration: $registration,
            mileage: $mileage,
            make: $valuation['make'] ?? '',
            model: $valuation['model'] ?? '',
            derivative: $valuation['derivativeName'] ?? '',
            fuelType: $valuation['fuelType'] ?? '',
            engineSize: $valuation['engineSize'] ?? '',
            bodyType: $valuation['bodyType'] ?? '',
            transmission: $valuation['transmission'] ?? '',
            registrationDate: Carbon::createFromFormat('d-m-Y', $valuation['dateOfRegistration']),
            vehicleType: $valuation['vehicleType'] ?? '',
            retailPrice: $valuation['valuation'],
        );
    }

    protected function headers(): array
    {
        return [
            'Authorization' => "Bearer " . $this->accessToken()
        ];
    }

    protected function accessToken(): string
    {
        $cacheKey = 'auto-trader-api-access-token-' . $this->config->apiKey();
        return Cache::remember(
            $cacheKey,
            Carbon::now()->addMinutes(15),
            fn() => Http::post('https://api.autotrader.co.uk/authenticate', [
                'key' => $this->config->apiKey(),
                'secret' => $this->config->apiSecret()
            ])
                ->onError(fn($error) => Log::error('AutoTrader API token error', [$error]))
                ->json('access_token')
        );
    }

    protected function endpoint(string $registration, int $mileage): string
    {
        return 'https://api.autotrader.co.uk/service/part-ex-valuation/v1/registration/'
            . $registration . '/mileage/' . $mileage;
    }
}
