<?php

namespace Mtc\VehicleValuation;

use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Config\AutoTraderConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;

class AutoTrader implements Contracts\ValuationDriver
{
    public function __construct(protected AutoTraderConfig $config)
    {
        //
    }

    public function getValuation(string $registration, int $mileage): ValuationResponse
    {
        $vehicleData = $this->getVehicleData($registration);
        $valuation = Http::withHeaders($this->headers())
            ->post($this->endpoint('service/stock-management/valuations?advertiserId=' . $this->config->merchantId()), [
                'vehicle' => [
                    "derivativeId" => $vehicleData['derivativeId'],
                    "firstRegistrationDate" => $vehicleData['firstRegistrationDate'],
                    "odometerReadingMiles" => $mileage,
                ],
                "conditionRating" => "Good",
            ])
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('AutoTrader Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);
                throw new ValuationRequestFailed('AutoTrader Valuation request failed');
            })->json('valuations');

        return new ValuationResponse(
            provider: 'auto-trader',
            registration: $registration,
            mileage: $mileage,
            vehicleType: $vehicleData['vehicleType'] ?? '',
            retailPrice: $valuation['trade']['amountGBP'],
            averagePrice: $valuation['private']['amountGBP'],
            cleanPrice: $valuation['retail']['amountGBP'],
            belowPrice: $valuation['partExchange']['amountGBP'],
            make: $vehicleData['make'] ?? '',
            model: $vehicleData['model'] ?? '',
            derivative: $vehicleData['derivative'] ?? '',
            fuelType: $vehicleData['fuelType'] ?? '',
            engineSize: $vehicleData['engineSize'] ?? '',
            bodyType: $vehicleData['bodyType'] ?? '',
            transmission: $vehicleData['transmission'] ?? '',
            registrationDate: Carbon::createFromFormat('Y-m-d', $vehicleData['firstRegistrationDate']),
        );
    }

    protected function headers(): array
    {
        return [
            'Authorization' => "Bearer " . $this->accessToken()
        ];
    }

    protected function accessToken(): string
    {
        $cacheKey = 'auto-trader-api-access-token-' . $this->config->apiKey();
        return Cache::remember(
            $cacheKey,
            Carbon::now()->addMinutes(15),
            fn() => Http::post($this->endpoint('authenticate'), [
                'key' => $this->config->apiKey(),
                'secret' => $this->config->apiSecret()
            ])
                ->onError(fn(Response $error) => Log::error('AutoTrader API token error', [
                    'endpoint' => $this->endpoint('authenticate'),
                    'response' => $error->body(),
                ]))
                ->json('access_token')
        );
    }

    protected function endpoint(string $path): string
    {
        return $this->config->live()
            ? 'https://api.autotrader.co.uk/' . $path
            : 'https://api-sandbox.autotrader.co.uk/' . $path;
    }

    /**
     * @param string $derivativeId
     * @param int $mileage
     * @return string
     */
    public function valuationEndpoint(string $derivativeId, int $mileage): string
    {
        $params = [
            'advertiserId' => $this->config->merchantId(),
            'derivativeId' => $derivativeId,
            'odometerReadingMiles' => $mileage,
        ];

        return $this->endpoint("service/stock-management/valuations?" . http_build_query($params));
    }

    /**
     * @param string $registration
     * @return array|null
     * @throws ValuationRequestFailed
     */
    public function getVehicleData(string $registration): ?array
    {
        return Http::withHeaders($this->headers())
            ->get($this->endpoint('service/stock-management/vehicles?' . http_build_query([
                    'registration' => $registration,
                    'advertiserId' => $this->config->merchantId(),
                ])))
            ->onError(function (Response $response) use ($registration) {
                Log::warning('AutoTrader Valuation failed to get derivative ID', [
                    'message' => $response->body(),
                    'registration' => $registration,
                    'advertiserId' => $this->config->merchantId(),
                ]);
                throw new ValuationRequestFailed('AutoTrader Valuation request failed: ' . $response->json('message'));
            })->json('vehicle');
    }

}
