<?php

namespace Mtc\VehicleValuation;

use App\Http\Requests\ValuationCustomerRequest;
use Carbon\Carbon;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\VehicleValuation\Models\VehicleValuation;
use Mtc\VehicleValuation\Config\AutoTraderConfig;
use Mtc\VehicleValuation\Contracts\ValuationResponse;
use Mtc\VehicleValuation\Exceptions\ValuationRequestFailed;

class AutoTrader implements Contracts\ValuationDriver, Contracts\HasFutureValuations
{
    public function __construct(protected AutoTraderConfig $config)
    {
        //
    }

    public function getValuation(string $registration, int $mileage, ?string $variant = null): ValuationResponse
    {
        $vehicleData = $this->getVehicleData($registration);
        $valuation = Http::withHeaders($this->headers())
            ->post($this->endpoint('valuations?advertiserId=' . $this->config->merchantId()), [
                'vehicle' => [
                    "derivativeId" => $vehicleData['derivativeId'],
                    "firstRegistrationDate" => $vehicleData['firstRegistrationDate'],
                    "odometerReadingMiles" => $mileage,
                ],
                "conditionRating" => "Good",
            ])
            ->onError(function (Response $response) use ($registration, $mileage) {
                Log::warning('AutoTrader Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $registration,
                    'mileage' => $mileage,
                ]);
                throw new ValuationRequestFailed('Failed to get valuation details from AutoTrader.');
            })->json('valuations');

        if (empty($valuation)) {
            throw new ValuationRequestFailed('Failed to get valuation details from AutoTrader');
        }

        return new ValuationResponse(
            provider: 'auto-trader',
            registration: $registration,
            mileage: $mileage,
            vehicleType: $vehicleData['vehicleType'] ?? '',
            retailPrice: $valuation['trade']['amountGBP'],
            averagePrice: $valuation['private']['amountGBP'],
            cleanPrice: $valuation['retail']['amountGBP'],
            belowPrice: $valuation['partExchange']['amountGBP'],
            make: $vehicleData['make'] ?? '',
            model: $vehicleData['model'] ?? '',
            derivative: $vehicleData['derivative'] ?? '',
            fuelType: $vehicleData['fuelType'] ?? '',
            engineSize: $vehicleData['engineSize'] ?? '',
            bodyType: $vehicleData['bodyType'] ?? '',
            transmission: $vehicleData['transmission'] ?? '',
            registrationDate: Carbon::createFromFormat('Y-m-d', $vehicleData['firstRegistrationDate']),
            colour: $vehicleData['colour'] ?? '',
            data: [
                "derivativeId" => $vehicleData['derivativeId'],
                "VIN" => $vehicleData['vin'],
                "previous_owners" => $vehicleData['owners'],
                "firstRegistrationDate" => $vehicleData['firstRegistrationDate']
            ]
        );
    }

    public function getFutureValuation(VehicleValuation $valuation, int $mileage_per_year, string $condition): array
    {
        $vehicleData = $valuation->data ?? $this->getVehicleData($valuation->registration);
        $future_value = Http::withHeaders($this->headers())
            ->post($this->endpoint('valuations/trends?advertiserId=' . $this->config->merchantId()), [
                'vehicle' => [
                    "derivativeId" => $vehicleData['derivativeId'],
                    "firstRegistrationDate" => $vehicleData['firstRegistrationDate'],
                ],
                'conditionRating' => $condition,
                'valuations' => [
                    'markets' => [
                        'retail',
                        'partExchange',
                        'trade',
                        'private',
                    ],
                    'frequency' => 'month',
                    'start' => [
                        'date' => Carbon::now()->format('Y-m-d'),
                        'odometerReadingMiles' => $valuation->mileage,
                    ],
                    'end' => [
                        'date' => Carbon::now()->addMonths(6)->format('Y-m-d'),
                        'odometerReadingMiles' => $valuation->mileage + ($mileage_per_year / 2),
                    ],
                ],
            ])
            ->onError(function (Response $response) use ($valuation) {
                Log::warning('AutoTrader Valuation Request failed', [
                    'message' => $response->body(),
                    'status' => $response->status(),
                    'registration' => $valuation->registration,
                    'mileage' => $valuation->mileage,
                ]);
                throw new ValuationRequestFailed('Failed to get valuation details from AutoTrader.');
            })->json('valuations');

        return collect($future_value)->keyBy('date')->map(fn($value) => [
            'retail' => $value['retail']['amountGBP'] ?? null,
            'partExchange' => $value['partExchange']['amountGBP'] ?? null,
            'trade' => $value['trade']['amountGBP'] ?? null,
            'private' => $value['private']['amountGBP'] ?? null,
        ])->toArray();

    }

    protected function headers(): array
    {
        return [
            'Authorization' => "Bearer " . $this->accessToken()
        ];
    }

    protected function accessToken(): string
    {
        $cacheKey = 'auto-trader-api-access-token-' . $this->config->apiKey();
        return Cache::remember(
            $cacheKey,
            Carbon::now()->addMinutes(15),
            fn() => Http::post($this->endpoint('authenticate'), [
                'key' => $this->config->apiKey(),
                'secret' => $this->config->apiSecret()
            ])
                ->onError(fn(Response $error) => Log::error('AutoTrader API token error', [
                    'endpoint' => $this->endpoint('authenticate'),
                    'response' => $error->body(),
                ]))
                ->json('access_token')
        );
    }

    protected function endpoint(string $path): string
    {
        return $this->config->live()
            ? 'https://api.autotrader.co.uk/' . $path
            : 'https://api-sandbox.autotrader.co.uk/' . $path;
    }

    /**
     * @param string $derivativeId
     * @param int $mileage
     * @return string
     */
    public function valuationEndpoint(string $derivativeId, int $mileage): string
    {
        $params = [
            'advertiserId' => $this->config->merchantId(),
            'derivativeId' => $derivativeId,
            'odometerReadingMiles' => $mileage,
        ];

        return $this->endpoint("valuations?" . http_build_query($params));
    }

    /**
     * @param string $registration
     * @return array|null
     * @throws ValuationRequestFailed
     */
    public function getVehicleData(string $registration, ?string $return = 'vehicle', array $params = []): ?array
    {
        return Http::withHeaders($this->headers())
            ->get($this->endpoint('vehicles?' . http_build_query(array_merge([
                    'registration' => $registration,
                    'advertiserId' => $this->config->merchantId(),
                ], $params))))
            ->onError(function (Response $response) use ($registration) {
                Log::warning('AutoTrader Valuation failed to get derivative ID', [
                    'message' => $response->body(),
                    'registration' => $registration,
                    'advertiserId' => $this->config->merchantId(),
                ]);
                throw new ValuationRequestFailed('AutoTrader was not able to find vehicle: ' . $response->json('message'));
            })->json($return);
    }

    public function sendCustomerData(ValuationCustomerRequest $request, VehicleValuation $valuation): bool
    {
        return true;
    }
}
