<?php
/**
 * Audit Logger
 *
 * Handles logging of authentication attempts to the database.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_Audit_Logger class.
 *
 * Logs authentication attempts for audit purposes.
 */
class MTC_Audit_Logger {

	/**
	 * Number of days to retain audit logs.
	 *
	 * @var int
	 */
	const RETENTION_DAYS = 7;

	/**
	 * Constructor.
	 *
	 * Registers WordPress hooks for audit logging.
	 */
	public function __construct() {
		// Register cleanup hook
		add_action( 'mtc_admin_auth_cleanup_logs', array( $this, 'cleanup_old_logs' ) );
	}

	/**
	 * Log an authentication attempt.
	 *
	 * Implements FR-034: Log all authentication attempts.
	 * Implements FR-035: Record timestamp, source email, domain, success status.
	 * Implements FR-036: Record error messages for failed attempts.
	 * T046: Core audit logging implementation.
	 * T047: Log successful authentications.
	 * T048: Log failed authentications.
	 *
	 * @param string      $email         Source email address.
	 * @param string      $domain        Current domain.
	 * @param bool        $success       Whether authentication succeeded.
	 * @param string|null $error_message Error message if failed.
	 * @return bool True on success, false on failure.
	 */
	public function log_attempt( $email, $domain, $success, $error_message = null ) {
		global $wpdb;

		$table_name = $wpdb->prefix . 'mtc_auth_logs';

		// Prepare log data (FR-034, FR-035, FR-036)
		$log_data = array(
			'timestamp'     => current_time( 'mysql' ),
			'source_email'  => sanitize_email( $email ),
			'domain'        => sanitize_text_field( $domain ),
			'success'       => $success ? 1 : 0,
			'ip_address'    => $this->get_ip_address(), // T049
			'user_agent'    => $this->get_user_agent(), // T050
			'error_message' => $error_message ? sanitize_text_field( $error_message ) : null,
		);

		$log_format = array(
			'%s', // timestamp
			'%s', // source_email
			'%s', // domain
			'%d', // success
			'%s', // ip_address
			'%s', // user_agent
			'%s', // error_message
		);

		// Insert log entry into database
		$result = $wpdb->insert( $table_name, $log_data, $log_format );

		if ( false === $result ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Failed to insert audit log - ' . $wpdb->last_error );
			}
			return false;
		}

		return true;
	}

	/**
	 * Get IP address of current request.
	 *
	 * Implements FR-037: Capture IP address for audit logs.
	 * T049: IP address capture implementation.
	 *
	 * @return string IP address.
	 */
	private function get_ip_address() {
		// Check common headers for IP address (in order of preference)
		$ip_headers = array(
			'HTTP_CF_CONNECTING_IP', // Cloudflare
			'HTTP_X_REAL_IP',        // Nginx proxy
			'HTTP_X_FORWARDED_FOR',  // Standard proxy header
			'REMOTE_ADDR',           // Direct connection
		);

		foreach ( $ip_headers as $header ) {
			if ( ! empty( $_SERVER[ $header ] ) ) {
				$ip = sanitize_text_field( wp_unslash( $_SERVER[ $header ] ) );

				// HTTP_X_FORWARDED_FOR can contain multiple IPs, use the first one
				if ( strpos( $ip, ',' ) !== false ) {
					$ip_list = explode( ',', $ip );
					$ip      = trim( $ip_list[0] );
				}

				// Validate IP address
				if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
					return $ip;
				}
			}
		}

		return '0.0.0.0';
	}

	/**
	 * Get user agent of current request.
	 *
	 * Implements FR-038: Capture user agent for audit logs.
	 * T050: User agent capture implementation.
	 *
	 * @return string User agent string.
	 */
	private function get_user_agent() {
		if ( ! empty( $_SERVER['HTTP_USER_AGENT'] ) ) {
			return sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) );
		}

		return '';
	}

	/**
	 * Clean up audit logs older than retention period.
	 *
	 * Deletes records older than RETENTION_DAYS (7 days by default).
	 * Runs automatically via WordPress cron.
	 *
	 * @return int|false Number of rows deleted, or false on error.
	 */
	public function cleanup_old_logs() {
		global $wpdb;

		$table_name = $wpdb->prefix . 'mtc_auth_logs';

		// Calculate cutoff date (7 days ago)
		$cutoff_date = gmdate( 'Y-m-d H:i:s', strtotime( '-' . self::RETENTION_DAYS . ' days' ) );

		// Delete old records
		$deleted = $wpdb->query(
			$wpdb->prepare(
				"DELETE FROM {$table_name} WHERE timestamp < %s",
				$cutoff_date
			)
		);

		if ( false !== $deleted && $deleted > 0 ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Cleaned up ' . $deleted . ' audit log records older than ' . self::RETENTION_DAYS . ' days' );
			}
		}

		return $deleted;
	}
}
