<?php
/**
 * OTP Validation Service
 *
 * Handles validation of OTP codes against the MTC authentication API.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_OTP_Service class.
 *
 * Validates OTP codes via external API.
 */
class MTC_OTP_Service {

	/**
	 * OTP validation API URL.
	 *
	 * @var string
	 */
	const API_URL = 'https://projects.mtcmedia.co.uk/api/vcs/admin-access-check';

	/**
	 * API request timeout in seconds.
	 *
	 * @var int
	 */
	const TIMEOUT = 10;

	/**
	 * Validate OTP code against external API.
	 *
	 * Makes GET request to MTC authentication API to validate the OTP.
	 * Implements comprehensive error handling for network and parsing failures.
	 *
	 * @param string $email  MTC staff email address.
	 * @param string $otp    One-time password code.
	 * @param string $domain Current domain.
	 * @return array Array with 'valid' (bool) and 'message' (string) keys.
	 */
	public function validate_otp( $email, $otp, $domain ) {
		// Build API URL with query parameters
		$url = add_query_arg(
			array(
				'email'  => urlencode( $email ),
				'otp'    => urlencode( $otp ),
				'domain' => urlencode( $domain ),
			),
			self::API_URL
		);

		// Make GET request with timeout
		$response = wp_remote_get(
			$url,
			array(
				'timeout'   => self::TIMEOUT,
				'sslverify' => true,
			)
		);

		// Handle connection errors (T025 - FR-022)
		if ( is_wp_error( $response ) ) {
			// Log error if debug mode enabled (T070)
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: API connection error - ' . $response->get_error_message() );
			}

			return array(
				'valid'   => false,
				'message' => __( 'Unable to reach authentication service. Please try again later.', 'mtc-admin-auth' ),
			);
		}

		// Check HTTP status code first (T060)
		// Only treat 5xx server errors as connection failures
		// 4xx client errors (like 403) may have valid JSON responses
		$status_code = wp_remote_retrieve_response_code( $response );
		if ( $status_code >= 500 ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: API returned server error HTTP ' . $status_code );
			}

			return array(
				'valid'   => false,
				'message' => __( 'Authentication service temporarily unavailable. Please try again later.', 'mtc-admin-auth' ),
			);
		}

		// Get response body
		$body = wp_remote_retrieve_body( $response );

		// Handle empty response
		if ( empty( $body ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Empty API response' );
			}

			return array(
				'valid'   => false,
				'message' => __( 'Authentication service error. Please contact support.', 'mtc-admin-auth' ),
			);
		}

		// Parse JSON response (T024 - FR-004)
		$data = json_decode( $body, true );

		// Handle JSON parsing errors (T025 - FR-023)
		if ( json_last_error() !== JSON_ERROR_NONE ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: JSON decode error - ' . json_last_error_msg() );
			}

			return array(
				'valid'   => false,
				'message' => __( 'Authentication service error. Please contact support.', 'mtc-admin-auth' ),
			);
		}

		// Validate response structure (T024)
		if ( ! isset( $data['valid'] ) || ! isset( $data['message'] ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Invalid API response structure' );
			}

			return array(
				'valid'   => false,
				'message' => __( 'Authentication service error. Please contact support.', 'mtc-admin-auth' ),
			);
		}

		// Return API response (T024, T025 - FR-004, FR-021)
		// The API tells us if auth succeeded via the 'valid' field
		// HTTP status codes like 403 are handled by returning the API's message
		return array(
			'valid'   => (bool) $data['valid'],
			'message' => sanitize_text_field( $data['message'] ),
		);
	}
}
