<?php
/**
 * Session Manager
 *
 * Handles session establishment, timeout, and metadata storage.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_Session_Manager class.
 *
 * Manages WordPress sessions for authenticated users.
 */
class MTC_Session_Manager {

	/**
	 * Session timeout in seconds (24 hours).
	 *
	 * @var int
	 */
	const SESSION_TIMEOUT = 86400;

	/**
	 * Constructor.
	 *
	 * Registers WordPress hooks for session management.
	 */
	public function __construct() {
		// T051: Register session timeout filter (FR-029)
		add_filter( 'auth_cookie_expiration', array( $this, 'filter_cookie_expiration' ), 10, 3 );
	}

	/**
	 * Establish WordPress session for user.
	 *
	 * Implements FR-011: Logs user into WordPress.
	 * Implements FR-012: Creates WordPress session.
	 *
	 * @param WP_User $user         User object.
	 * @param string  $source_email MTC staff email address.
	 * @return bool True on success, false on failure.
	 */
	public function establish_session( $user, $source_email ) {
		if ( ! $user instanceof WP_User || ! $user->ID ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Invalid user object for session establishment' );
			}
			return false;
		}

		// Set current user (FR-011)
		wp_set_current_user( $user->ID, $user->user_login );

		// Set authentication cookie with 24-hour expiration (FR-012)
		wp_set_auth_cookie( $user->ID, false );

		// Fire WordPress login action hook
		// This allows other plugins to react to the login event
		do_action( 'wp_login', $user->user_login, $user );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: Session established for user ID ' . $user->ID . ' (source: ' . $source_email . ')' );
		}

		return true;
	}

	/**
	 * Update user metadata with authentication info.
	 *
	 * Implements FR-013: Stores source email in user metadata.
	 * Implements FR-014: Records login timestamp.
	 * Implements FR-027: Updates login count.
	 * Implements FR-028: Stores session expiration time.
	 * Implements FR-029: 24-hour session duration.
	 *
	 * @param int    $user_id      User ID.
	 * @param string $source_email MTC staff email address.
	 * @return bool True on success, false on failure.
	 */
	public function update_metadata( $user_id, $source_email ) {
		if ( ! $user_id || ! is_numeric( $user_id ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Invalid user ID for metadata update' );
			}
			return false;
		}

		// FR-013: Store the source email address
		update_user_meta( $user_id, 'mtcadmin_source_email', sanitize_email( $source_email ) );

		// FR-014, FR-027: Update last login timestamp
		$current_time = current_time( 'mysql' );
		update_user_meta( $user_id, 'last_login', $current_time );

		// FR-027: Increment login count
		$login_count = (int) get_user_meta( $user_id, 'mtcadmin_login_count', true );
		update_user_meta( $user_id, 'mtcadmin_login_count', $login_count + 1 );

		// FR-028, FR-029: Store session expiration (24 hours from now)
		$session_expires = time() + self::SESSION_TIMEOUT;
		update_user_meta( $user_id, 'mtcadmin_session_expires', $session_expires );

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: Updated metadata for user ID ' . $user_id . ' (login count: ' . ( $login_count + 1 ) . ')' );
		}

		return true;
	}

	/**
	 * Filter authentication cookie expiration.
	 *
	 * Implements FR-029: Enforce 24-hour session timeout for mtcadmin.
	 * T051: Register session timeout filter.
	 * T052: Implement session timeout logic.
	 *
	 * @param int  $expiration Expiration time in seconds.
	 * @param int  $user_id    User ID.
	 * @param bool $remember   Whether to remember the login.
	 * @return int Filtered expiration time.
	 */
	public function filter_cookie_expiration( $expiration, $user_id, $remember ) {
		// Get user data
		$user = get_userdata( $user_id );

		// Only apply to mtcadmin user
		if ( ! $user || $user->user_login !== MTC_User_Manager::USERNAME ) {
			return $expiration;
		}

		// Enforce 24-hour session timeout for mtcadmin (FR-029)
		// Ignore the $remember parameter - mtcadmin sessions always expire after 24 hours
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: Applying 24-hour session timeout for mtcadmin user' );
		}

		return self::SESSION_TIMEOUT;
	}
}
