<?php
/**
 * Plugin Name: MTC Admin Authentication
 * Plugin URI: https://www.mtc.co.uk
 * Description: OTP-based authentication for MTC staff
 * Version: 1.0.0
 * Author: MTC Media
 * Author URI: https://www.mtc.co.uk
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: mtc-admin-auth
 * Domain Path: /languages
 * Requires at least: 5.0
 * Requires PHP: 7.4
 */

// If this file is called directly, abort.
if ( ! defined( 'WPINC' ) ) {
	die;
}

/**
 * Current plugin version.
 */
define( 'MTC_ADMIN_AUTH_VERSION', '1.0.0' );

/**
 * Plugin directory path.
 */
define( 'MTC_ADMIN_AUTH_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

/**
 * Plugin directory URL.
 */
define( 'MTC_ADMIN_AUTH_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Code that runs during plugin activation.
 *
 * Creates the audit log database table and mtcadmin user.
 * Implements FR-017: Create mtcadmin user on activation.
 */
function activate_mtc_admin_auth() {
	global $wpdb;

	// Create audit log database table
	$table_name      = $wpdb->prefix . 'mtc_auth_logs';
	$charset_collate = $wpdb->get_charset_collate();

	$sql = "CREATE TABLE $table_name (
		id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
		timestamp datetime NOT NULL,
		source_email varchar(255) NOT NULL,
		domain varchar(255) NOT NULL,
		success tinyint(1) NOT NULL,
		ip_address varchar(45) NOT NULL,
		user_agent text,
		error_message text,
		PRIMARY KEY  (id),
		KEY idx_timestamp (timestamp),
		KEY idx_email (source_email),
		KEY idx_success (success),
		KEY idx_domain (domain)
	) $charset_collate;";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql );

	// Load user manager class to create mtcadmin user (FR-017)
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-user-manager.php';
	$user_manager = new MTC_User_Manager();

	// Check if mtcadmin user already exists
	$existing_user = $user_manager->get_mtcadmin_user();

	if ( ! $existing_user ) {
		// Create mtcadmin user
		$result = $user_manager->create_mtcadmin_user();

		if ( is_wp_error( $result ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Failed to create mtcadmin user during activation - ' . $result->get_error_message() );
			}
		} else {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Successfully created mtcadmin user during activation' );
			}
		}
	} else {
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: mtcadmin user already exists during activation' );
		}
	}

	// Schedule daily cleanup of old audit logs (older than 7 days)
	if ( ! wp_next_scheduled( 'mtc_admin_auth_cleanup_logs' ) ) {
		wp_schedule_event( time(), 'daily', 'mtc_admin_auth_cleanup_logs' );
	}
}

/**
 * Code that runs during plugin deactivation.
 *
 * Regenerates mtcadmin password for security.
 * Implements FR-018: Reset password on deactivation.
 *
 * Note: Audit logs are preserved. Use uninstall.php for cleanup.
 */
function deactivate_mtc_admin_auth() {
	// Load user manager class to reset mtcadmin password (FR-018)
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-user-manager.php';
	$user_manager = new MTC_User_Manager();

	// Get mtcadmin user
	$mtcadmin_user = $user_manager->get_mtcadmin_user();

	if ( $mtcadmin_user ) {
		// Regenerate password for security
		$result = $user_manager->regenerate_password( $mtcadmin_user->ID );

		if ( $result ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Successfully regenerated mtcadmin password during deactivation' );
			}
		} else {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Failed to regenerate mtcadmin password during deactivation' );
			}
		}
	} else {
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: mtcadmin user not found during deactivation' );
		}
	}

	// Clear scheduled cron events
	$timestamp = wp_next_scheduled( 'mtc_admin_auth_cleanup_logs' );
	if ( $timestamp ) {
		wp_unschedule_event( $timestamp, 'mtc_admin_auth_cleanup_logs' );
	}

	// Audit logs are intentionally preserved on deactivation
	// Use uninstall.php to remove data if plugin is deleted
}

register_activation_hook( __FILE__, 'activate_mtc_admin_auth' );
register_deactivation_hook( __FILE__, 'deactivate_mtc_admin_auth' );

/**
 * Load plugin dependencies.
 *
 * Include all required class files.
 */
function mtc_admin_auth_load_dependencies() {
	// Load core classes
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-auth-handler.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-otp-service.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-user-manager.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-security.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-audit-logger.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-session-manager.php';
	require_once MTC_ADMIN_AUTH_PLUGIN_DIR . 'includes/class-mtc-plugin-updater.php';
}

/**
 * Initialize the plugin.
 *
 * Instantiate the auth handler on init hook (WordPress 6.7+ requirement).
 */
function mtc_admin_auth_init() {
	// Instantiate the main authentication handler
	// This will automatically register WordPress hooks via its constructor
	new MTC_Auth_Handler();

	// Initialize plugin updater (admin only)
	if ( is_admin() ) {
		new MTC_Plugin_Updater(
			__FILE__,
			'https://composer.mtcassets.com/p2/mtcmedia/wordpress-mtc-admin-auth.json'
		);
	}
}

/**
 * Begin execution of the plugin.
 *
 * Load dependencies and initialize the plugin.
 */
function run_mtc_admin_auth() {
	// Load plugin dependencies
	mtc_admin_auth_load_dependencies();

	// Initialize plugin on init hook (WordPress 6.7+ requirement)
	// This ensures translations are loaded before any __() calls
	add_action( 'init', 'mtc_admin_auth_init' );
}

// Start the plugin
run_mtc_admin_auth();
