<?php
/**
 * Security Handler
 *
 * Handles CSRF protection, redirect validation, and password recovery blocking.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_Security class.
 *
 * Provides security controls for the authentication system.
 */
class MTC_Security {

	/**
	 * Nonce action name for CSRF protection.
	 *
	 * @var string
	 */
	const NONCE_ACTION = 'mtc_admin_auth_login';

	/**
	 * Nonce field name.
	 *
	 * @var string
	 */
	const NONCE_FIELD = 'mtc_auth_nonce';

	/**
	 * Constructor.
	 *
	 * Registers WordPress hooks for security features.
	 */
	public function __construct() {
		// Register nonce field on login form (T042)
		add_action( 'login_form', array( $this, 'add_nonce_field' ) );
	}

	/**
	 * Add CSRF nonce field to login form.
	 *
	 * Implements FR-032: Generate CSRF token for login form.
	 * T040: Generate nonce for CSRF protection.
	 */
	public function add_nonce_field() {
		// Output nonce field with our custom action and field name
		wp_nonce_field( self::NONCE_ACTION, self::NONCE_FIELD, true, true );
	}

	/**
	 * Validate CSRF nonce.
	 *
	 * Implements FR-033: Validate CSRF token on login.
	 * T041: Validate nonce to prevent CSRF attacks.
	 * T044: Error handling for invalid nonces.
	 *
	 * @return bool True if valid, false otherwise.
	 */
	public function validate_nonce() {
		// Check if nonce field is present
		if ( ! isset( $_POST[ self::NONCE_FIELD ] ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: CSRF nonce field missing' );
			}
			return false;
		}

		// Verify nonce
		$nonce = sanitize_text_field( wp_unslash( $_POST[ self::NONCE_FIELD ] ) );
		$valid = wp_verify_nonce( $nonce, self::NONCE_ACTION );

		if ( ! $valid ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: CSRF nonce validation failed' );
			}
			return false;
		}

		return true;
	}

	/**
	 * Validate redirect URL.
	 *
	 * Implements FR-039: Validate redirect URLs to prevent open redirects.
	 * T063: Implement redirect validation for security.
	 *
	 * @param string $redirect_to Redirect URL.
	 * @return string Validated redirect URL (admin_url() if invalid).
	 */
	public function validate_redirect( $redirect_to ) {
		// If redirect URL is empty, use admin dashboard
		if ( empty( $redirect_to ) ) {
			return admin_url();
		}

		// Parse the redirect URL
		$parsed_url = wp_parse_url( $redirect_to );

		// Check if URL has a host (external URL)
		if ( isset( $parsed_url['host'] ) ) {
			// Get current site URL
			$site_url = wp_parse_url( home_url() );

			// Only allow redirects to the same host
			if ( $parsed_url['host'] !== $site_url['host'] ) {
				if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
					error_log( 'MTC Admin Auth: Blocked external redirect to ' . $redirect_to );
				}
				return admin_url();
			}
		}

		// Use WordPress's wp_validate_redirect for additional security
		$validated_url = wp_validate_redirect( $redirect_to, admin_url() );

		return $validated_url;
	}
}
