<?php
/**
 * Main Authentication Handler
 *
 * Orchestrates the OTP authentication flow by coordinating between
 * OTP service, user manager, security, and session manager.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_Auth_Handler class.
 *
 * Handles authentication interception and orchestration.
 */
class MTC_Auth_Handler {

	/**
	 * OTP service instance.
	 *
	 * @var MTC_OTP_Service
	 */
	private $otp_service;

	/**
	 * User manager instance.
	 *
	 * @var MTC_User_Manager
	 */
	private $user_manager;

	/**
	 * Security handler instance.
	 *
	 * @var MTC_Security
	 */
	private $security;

	/**
	 * Session manager instance.
	 *
	 * @var MTC_Session_Manager
	 */
	private $session_manager;

	/**
	 * Audit logger instance.
	 *
	 * @var MTC_Audit_Logger
	 */
	private $audit_logger;

	/**
	 * Constructor.
	 *
	 * Initializes dependencies and registers WordPress hooks.
	 */
	public function __construct() {
		// Dependencies will be injected here
		$this->otp_service     = new MTC_OTP_Service();
		$this->user_manager    = new MTC_User_Manager();
		$this->security        = new MTC_Security();
		$this->session_manager = new MTC_Session_Manager();
		$this->audit_logger    = new MTC_Audit_Logger();

		// Block 'mtcadmin' username at priority 5 (before everything else)
		add_filter( 'authenticate', array( $this, 'block_mtcadmin_username' ), 5, 3 );

		// Register authentication filter with priority 10 (before WP default auth at 20)
		// This ensures we intercept MTC logins BEFORE password authentication is attempted
		add_filter( 'authenticate', array( $this, 'authenticate' ), 10, 3 );

		// Register login redirect filter
		add_filter( 'login_redirect', array( $this, 'handle_login_redirect' ), 10, 3 );
	}

	/**
	 * Block direct login attempts using 'mtcadmin' username.
	 *
	 * Runs at priority 5 to block before any authentication logic.
	 * T037, T038: Block password-based mtcadmin login (FR-030).
	 *
	 * @param WP_User|WP_Error|null $user     User object or error.
	 * @param string                $username Username submitted.
	 * @param string                $password Password submitted.
	 * @return WP_User|WP_Error|null WP_Error if 'mtcadmin' username, otherwise $user.
	 */
	public function block_mtcadmin_username( $user, $username, $password ) {
		// Check if username is 'mtcadmin' (case-insensitive)
		$username_lower = strtolower( trim( $username ) );
		if ( 'mtcadmin' === $username_lower ) {
			// Remove WordPress default authentication to prevent our error from being overridden
			remove_filter( 'authenticate', 'wp_authenticate_username_password', 20 );
			remove_filter( 'authenticate', 'wp_authenticate_email_password', 20 );

			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Blocked direct login attempt with mtcadmin username' );
			}

			return new WP_Error(
				'mtcadmin_direct_login_blocked',
				__( '<strong>Direct login as mtcadmin is not permitted.</strong><br>Please use your MTC email address (e.g., you@mtc.co.uk) instead of "mtcadmin" as the username.', 'mtc-admin-auth' )
			);
		}

		return $user;
	}

	/**
	 * Main authentication method.
	 *
	 * Intercepts WordPress authentication and handles OTP validation.
	 * Orchestrates the entire authentication flow.
	 *
	 * @param WP_User|WP_Error|null $user     User object or error.
	 * @param string                $username Username submitted.
	 * @param string                $password Password submitted (OTP code).
	 * @return WP_User|WP_Error|null User object on success, WP_Error on failure, null to pass through.
	 */
	public function authenticate( $user, $username, $password ) {

		// Check if we should handle this login attempt
		if ( ! $this->should_handle_login( $username ) ) {
			// Not an MTC login, pass through to WordPress default authentication
			return $user;
		}

		// This is an MTC login - remove WordPress default authentication to prevent override
		// We must handle this exclusively with OTP, so disable password-based auth
		remove_filter( 'authenticate', 'wp_authenticate_username_password', 20 );
		remove_filter( 'authenticate', 'wp_authenticate_email_password', 20 );

		// This is an MTC email login - we handle it exclusively with OTP authentication.
		// Note: 'mtcadmin' username is blocked separately at priority 5.

		// T043, T044: Validate CSRF nonce (FR-032, FR-033)
		if ( ! $this->security->validate_nonce() ) {
			return new WP_Error(
				'invalid_nonce',
				__( 'Security verification failed. Please refresh the page and try again.', 'mtc-admin-auth' )
			);
		}

		// Validate required fields
		if ( empty( $username ) || empty( $password ) ) {
			return new WP_Error(
				'empty_credentials',
				__( 'Please enter both email address and OTP code.', 'mtc-admin-auth' )
			);
		}

		// Get current domain for API validation
		$domain = $this->get_current_domain();

		if ( empty( $domain ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Unable to determine current domain' );
			}
			return new WP_Error(
				'domain_error',
				__( 'Unable to determine site domain. Please contact support.', 'mtc-admin-auth' )
			);
		}

		// Validate OTP with external API
		$validation_result = $this->otp_service->validate_otp( $username, $password, $domain );

		if ( ! $validation_result['valid'] ) {
			// Log to error log for debugging
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: OTP validation failed for ' . $username . ' - ' . $validation_result['message'] );
			}

			// T048: Log failed authentication attempt to database (FR-034, FR-036)
			$this->audit_logger->log_attempt(
				$username,
				$domain,
				false,
				$validation_result['message']
			);

			// OTP validation failed
			return new WP_Error(
				'invalid_otp',
				'MTC Admin Auth: ' . $validation_result['message']
			);
		}

		// OTP is valid, get or create mtcadmin user
		$mtcadmin_user = $this->user_manager->get_mtcadmin_user();

		if ( ! $mtcadmin_user ) {
			// User doesn't exist, create it
			$mtcadmin_user = $this->user_manager->create_mtcadmin_user();

			if ( is_wp_error( $mtcadmin_user ) ) {
				return new WP_Error(
					'user_creation_failed',
					__( 'MTC Admin Auth: Unable to create admin user. Please contact support.', 'mtc-admin-auth' )
				);
			}
		}

		// Regenerate password for security
		$password_regenerated = $this->user_manager->regenerate_password( $mtcadmin_user->ID );

		if ( ! $password_regenerated ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Password regeneration failed for user ID ' . $mtcadmin_user->ID );
			}
			// Continue anyway - not critical for authentication
		}

		// Establish WordPress session
		$session_established = $this->session_manager->establish_session( $mtcadmin_user, $username );

		if ( ! $session_established ) {
			return new WP_Error(
				'session_error',
				__( 'Unable to establish session. Please try again.', 'mtc-admin-auth' )
			);
		}

		// Update user metadata
		$metadata_updated = $this->session_manager->update_metadata( $mtcadmin_user->ID, $username );

		if ( ! $metadata_updated ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Metadata update failed for user ID ' . $mtcadmin_user->ID );
			}
			// Continue anyway - not critical for authentication
		}

		// T047: Log successful authentication attempt (FR-034, FR-035)
		$this->audit_logger->log_attempt(
			$username,
			$domain,
			true,
			null
		);

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: Authentication successful for ' . $username . ' on domain ' . $domain );
		}

		// Return authenticated user
		return $mtcadmin_user;
	}

	/**
	 * Check if this login attempt should be handled by OTP authentication.
	 *
	 * Detects MTC email addresses (@mtc.co.uk or @mtcmedia.co.uk).
	 * Note: 'mtcadmin' username is blocked separately and not handled here.
	 *
	 * @param string $username Username to check.
	 * @return bool True if should handle, false otherwise.
	 */
	private function should_handle_login( $username ) {
		if ( empty( $username ) ) {
			return false;
		}

		// Convert to lowercase for case-insensitive comparison
		$username_lower = strtolower( trim( $username ) );

		// Only handle MTC email addresses, not 'mtcadmin' username
		// 'mtcadmin' is blocked separately at priority 5
		// Validates both @mtc.co.uk and @mtcmedia.co.uk (legacy)
		if ( strpos( $username_lower, '@' ) !== false ) {
			return ( substr( $username_lower, -10 ) === '@mtc.co.uk' ||
			         substr( $username_lower, -15 ) === '@mtcmedia.co.uk' );
		}

		return false;
	}

	/**
	 * Extract current domain from HTTP_HOST header.
	 *
	 * Removes www. prefix and sanitizes the domain.
	 *
	 * @return string Current domain with www. prefix removed.
	 */
	private function get_current_domain() {
		// Get HTTP_HOST from server variables
		$host = isset( $_SERVER['HTTP_HOST'] ) ? $_SERVER['HTTP_HOST'] : '';

		if ( empty( $host ) ) {
			return '';
		}

		// Remove www. prefix (case-insensitive)
		$domain = preg_replace( '/^www\./i', '', $host );

		// Sanitize domain - remove any non-domain characters
		$domain = preg_replace( '/[^a-z0-9.-]/i', '', $domain );

		return $domain;
	}

	/**
	 * Handle login redirect after successful authentication.
	 *
	 * Redirects mtcadmin user to admin dashboard with success message.
	 * Implements FR-015: Redirect to dashboard.
	 * Implements FR-016: Display success message.
	 *
	 * @param string  $redirect_to           URL to redirect to.
	 * @param string  $requested_redirect_to Original redirect destination.
	 * @param WP_User $user                  User object.
	 * @return string Filtered redirect URL.
	 */
	public function handle_login_redirect( $redirect_to, $requested_redirect_to, $user ) {
		// Check if this is a valid user and it's the mtcadmin user
		if ( ! isset( $user->user_login ) || $user->user_login !== MTC_User_Manager::USERNAME ) {
			// Not mtcadmin, use default redirect
			return $redirect_to;
		}

		// Get source email from user metadata
		$source_email = get_user_meta( $user->ID, 'mtcadmin_source_email', true );

		// Redirect to admin dashboard with success message (FR-015, FR-016)
		$redirect_url = admin_url();
		$redirect_url = add_query_arg( 'mtc_login', 'success', $redirect_url );

		if ( ! empty( $source_email ) ) {
			$redirect_url = add_query_arg( 'mtc_email', urlencode( $source_email ), $redirect_url );
		}

		return $redirect_url;
	}
}
