<?php
/**
 * Plugin Updater
 *
 * Handles plugin updates from custom repository instead of WordPress.org
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_Plugin_Updater class.
 *
 * Checks for plugin updates from custom repository.
 */
class MTC_Plugin_Updater {

	/**
	 * Plugin slug.
	 *
	 * @var string
	 */
	private $plugin_slug;

	/**
	 * Plugin basename.
	 *
	 * @var string
	 */
	private $plugin_basename;

	/**
	 * Current plugin version.
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Update API URL.
	 *
	 * @var string
	 */
	private $api_url;

	/**
	 * Constructor.
	 *
	 * @param string $plugin_file     Main plugin file path.
	 * @param string $api_url         URL to check for updates.
	 */
	public function __construct( $plugin_file, $api_url ) {
		$this->plugin_slug     = dirname( plugin_basename( $plugin_file ) );
		$this->plugin_basename = plugin_basename( $plugin_file );
		$this->api_url         = $api_url;

		// Get plugin data
		if ( ! function_exists( 'get_plugin_data' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		$plugin_data   = get_plugin_data( $plugin_file );
		$this->version = $plugin_data['Version'];

		// Hook into WordPress update checks
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_update' ) );
		add_filter( 'plugins_api', array( $this, 'plugin_info' ), 10, 3 );
	}

	/**
	 * Check for plugin updates.
	 *
	 * @param object $transient Update transient object.
	 * @return object Modified transient.
	 */
	public function check_for_update( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		// Get update info from API
		$update_info = $this->get_update_info();

		if ( false === $update_info ) {
			return $transient;
		}

		// Check if new version is available
		if ( version_compare( $this->version, $update_info->version, '<' ) ) {
			$transient->response[ $this->plugin_basename ] = (object) array(
				'slug'        => $this->plugin_slug,
				'new_version' => $update_info->version,
				'package'     => $update_info->download_url,
				'url'         => $update_info->homepage,
				'tested'      => $update_info->tested,
				'requires'    => $update_info->requires,
				'requires_php' => $update_info->requires_php,
			);
		}

		return $transient;
	}

	/**
	 * Provide plugin information for update details screen.
	 *
	 * @param false|object|array $result The result object or array.
	 * @param string             $action The type of information being requested.
	 * @param object             $args   Plugin API arguments.
	 * @return false|object Modified result.
	 */
	public function plugin_info( $result, $action, $args ) {
		if ( 'plugin_information' !== $action ) {
			return $result;
		}

		if ( $this->plugin_slug !== $args->slug ) {
			return $result;
		}

		$update_info = $this->get_update_info();

		if ( false === $update_info ) {
			return $result;
		}

		return (object) array(
			'name'          => $update_info->name,
			'slug'          => $this->plugin_slug,
			'version'       => $update_info->version,
			'author'        => $update_info->author,
			'homepage'      => $update_info->homepage,
			'requires'      => $update_info->requires,
			'tested'        => $update_info->tested,
			'requires_php'  => $update_info->requires_php,
			'download_link' => $update_info->download_url,
			'sections'      => array(
				'description' => $update_info->description,
				'changelog'   => $update_info->changelog,
			),
		);
	}

	/**
	 * Get update information from Satis API.
	 *
	 * @return object|false Update info object or false on failure.
	 */
	private function get_update_info() {
		// Check cache first
		$cache_key = 'mtc_plugin_update_' . $this->plugin_slug;
		$cached    = get_transient( $cache_key );

		if ( false !== $cached ) {
			return $cached;
		}

		// Request update info from Satis p2 API
		$response = wp_remote_get(
			$this->api_url,
			array(
				'timeout' => 10,
				'headers' => array(
					'Accept' => 'application/json',
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Plugin Updater: Failed to check for updates - ' . $response->get_error_message() );
			}
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( empty( $data ) || ! isset( $data['packages'] ) ) {
			return false;
		}

		// Get package name from composer.json format
		$package_name = 'mtcmedia/wordpress-mtc-admin-auth';

		if ( ! isset( $data['packages'][ $package_name ] ) ) {
			return false;
		}

		$packages = $data['packages'][ $package_name ];

		// Find the latest stable version (not dev-*)
		$latest = null;
		$latest_version = '0.0.0';

		foreach ( $packages as $package ) {
			// Skip dev versions
			if ( strpos( $package['version'], 'dev-' ) === 0 ) {
				continue;
			}

			// Clean version string (remove 'v' prefix if present)
			$clean_version = ltrim( $package['version'], 'v' );

			if ( version_compare( $clean_version, $latest_version, '>' ) ) {
				$latest_version = $clean_version;
				$latest = $package;
			}
		}

		if ( ! $latest ) {
			return false;
		}

		// Convert Satis package data to WordPress update format
		$update_info = (object) array(
			'name'         => isset( $latest['description'] ) ? $latest['description'] : 'MTC Admin Authentication',
			'slug'         => $this->plugin_slug,
			'version'      => $latest_version,
			'author'       => isset( $latest['authors'][0]['name'] ) ? $latest['authors'][0]['name'] : 'MTC Media',
			'homepage'     => isset( $latest['homepage'] ) ? $latest['homepage'] : 'https://www.mtc.co.uk',
			'requires'     => '5.0',
			'tested'       => '6.4',
			'requires_php' => isset( $latest['require']['php'] ) ? str_replace( '>=', '', $latest['require']['php'] ) : '7.4',
			'download_url' => isset( $latest['dist']['url'] ) ? $latest['dist']['url'] : '',
			'description'  => isset( $latest['description'] ) ? '<p>' . esc_html( $latest['description'] ) . '</p>' : '',
			'changelog'    => $this->get_changelog( $latest_version ),
		);

		// Cache for 12 hours
		set_transient( $cache_key, $update_info, 12 * HOUR_IN_SECONDS );

		return $update_info;
	}

	/**
	 * Generate changelog HTML from version.
	 *
	 * @param string $version Version number.
	 * @return string Changelog HTML.
	 */
	private function get_changelog( $version ) {
		// Basic changelog - can be enhanced by reading CHANGELOG.md from repo
		return '<h4>' . esc_html( $version ) . '</h4><ul><li>See release notes on Bitbucket</li></ul>';
	}

}
