<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Controller\Adminhtml\Configuration;

use Laminas\Uri\Uri;
use Magento\Backend\App\Action\Context;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Escaper;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\State\UserLockedException;
use Magento\Framework\HTTP\ZendClient;
use Magento\Framework\Oauth\Exception as OauthException;
use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Integration\Api\OauthServiceInterface;
use Magento\Integration\Controller\Adminhtml\Integration as IntegrationController;
use Magento\Integration\Helper\Oauth\Data as IntegrationOauthHelper;
use Magento\Integration\Model\Integration as IntegrationModel;
use Magento\Integration\Model\Oauth\TokenFactory as TokenFactory;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Magento\Integration\Model\ConfigBasedIntegrationManager;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Zend_Http_Client;
use Magento\Framework\App\Config\Storage\WriterInterface;

class CreatePermissions extends Base implements HttpPostActionInterface
{
    const ADMIN_RESOURCE = 'Novuna_Pbf::paybyfinance';
    private PbfLoggerInterface $logger;
    private Escaper $escaper;
    private ConfigBasedIntegrationManager $integrationManager;
    private OauthServiceInterface $oauthService;
    private IntegrationServiceInterface $integrationService;
    private StoreConfigProvider $storeConfigProvider;
    private ZendClient $httpClient;
    private TokenFactory $tokenFactory;
    private StoreManagerInterface $storeManager;
    private IntegrationOauthHelper $dataHelper;
    private Uri $uri;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private WriterInterface $writerInterface;

    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        StoreConfigProvider $storeConfigProvider,
        OauthServiceInterface $oauthService,
        IntegrationServiceInterface $integrationService,
        ConfigBasedIntegrationManager $integrationManager,
        ZendClient $httpClient,
        StoreManagerInterface $storeManager,
        IntegrationOauthHelper $dataHelper,
        TokenFactory $tokenFactory,
        Escaper $escaper,
        PbfLoggerInterface $logger,
        Uri $uri,
        Context $context,
        WriterInterface $writerInterface
    ) {
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->oauthService = $oauthService;
        $this->integrationService = $integrationService;
        $this->integrationManager = $integrationManager;
        $this->escaper = $escaper;
        $this->logger = $logger;
        $this->httpClient = $httpClient;
        $this->tokenFactory = $tokenFactory;
        $this->storeManager = $storeManager;
        $this->dataHelper = $dataHelper;
        $this->uri = $uri;
        $this->writerInterface = $writerInterface;
        parent::__construct($context);
    }

    public function execute()
    {
        try {
            if ($this->isValidPostRequest()) {
                list($scope, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();
                $this->savePbfInstallationHmac(bin2hex(random_bytes(25)), $scope, $scopeId);

                if ($this->pbfWidgetConfigProvider->isPBFIntegrationCreated()) {
                    $params =  [IntegrationController::PARAM_INTEGRATION_ID => $this->pbfWidgetConfigProvider->getPBFIntegrationId()];
                    if ($scope != '') {
                        $params[$scope] = $scopeId;
                    }
                    return $this->_redirect(
                        'paybyfinance/configuration/installation',
                        $params
                    );
                }
                $this->writerInterface->save(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE, $scope);

                $this->_getSession()->setIntegrationData(null);
                $this->validateUser();
                $this->getRequest()->setParam('current_password', null);
                $integrationName = StoreConfigProvider::PBF_INTEGRATION_IDENTIFIER;
                $this->integrationManager->processIntegrationConfig([$integrationName]);

                $integration = $this->integrationService->findByName($integrationName);
                $integration->setEndpoint($this->getAppserverOauthUrl());
                /** Remove existing token associated with consumer before issuing a new one. */
                $this->oauthService->deleteIntegrationToken($integration->getConsumerId());
                $integration->setStatus(IntegrationModel::STATUS_INACTIVE)->save();
                
                $response = $this->postToConsumer(
                    $integration->getConsumerId(),
                    $integration->getEndpoint(),
                    $integration->getId()
                );

                $parsedUri = $this->uri->parse($response);
                $params = $parsedUri->getQueryAsArray();
                $params['id'] = $integration->getId();

                $this->_forward(
                    'permissionsdialog',
                    null,
                    null,
                    $params
                );
                return;
            }
        } catch (UserLockedException $e) {
            $this->_auth->logout();
            $this->_redirect('*');
        } catch (AuthenticationException $e) {
            $this->logger->error($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($e->getMessage());
        } catch (LocalizedException $e) {
            $this->logger->error($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($this->escaper->escapeHtml($e->getMessage()));
        } catch (\Exception $e) {
            $this->logger->critical($e . '\n' . $e->getTraceAsString());
            $this->messageManager->addErrorMessage($this->escaper->escapeHtml($e->getMessage()));
        }
        $this->redirect();
    }

    private function getAppserverOauthUrl(): string
    {
        return $this->pbfWidgetConfigProvider->getAppServerUrl() . PbfWidgetConfigProvider::PROXY_DESTINATION_PATH
            . 'oauthStart';
    }

    /**
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     * @throws Exception
     * @throws LocalizedException
     */
    public function postToConsumer($consumerId, $endpointUrl, $integrationID)
    {
        try {
            $consumer = $this->oauthService->loadConsumer($consumerId);
            if (!$consumer->getId()) {
                throw new OauthException(
                    __('A consumer with "%1" ID doesn\'t exist. Verify the ID and try again.', $consumerId)
                );
            }

            $consumerData = $consumer->getData();
            $verifier = $this->tokenFactory->create()->createVerifierToken($consumerId);
            $storeId = $this->_request->getParam('store');
            $storeBaseUrl = $this->storeManager->getStore($storeId)->getBaseUrl();
            $defaultStore = $this->storeManager->getDefaultStoreView();
            $storeCode = "";//$this->storeManager->getStore($storeId)->getCode();

            $this->httpClient->setUri($endpointUrl);
            $this->httpClient->setParameterGet("shop", $this->pbfWidgetConfigProvider->getStoreReference($defaultStore, false));
            $this->httpClient->setParameterGet("integrationId", $integrationID);
            $this->httpClient->setParameterGet("storeCode", $storeCode);
            // $this->httpClient->setParameterGet("consumerKey", $consumerData['key']);
            $this->httpClient->setParameterPost(
                [
                    'oauth_consumer_key' => $consumerData['key'],
                    'oauth_consumer_secret' => $consumerData['secret'],
                    'store_base_url' => $storeBaseUrl,
                    'oauth_verifier' => $verifier->getVerifier(),
                ]
            );
            $maxRedirects = $this->dataHelper->getConsumerPostMaxRedirects();
            $timeout = $this->dataHelper->getConsumerPostTimeout();
            $this->httpClient->setConfig(['maxredirects' => $maxRedirects, 'timeout' => $timeout]);
            $resp = $this->httpClient->request(Zend_Http_Client::POST);

            if ($resp->getStatus() != 302) {
                throw new LocalizedException(
                    __("Appserver did not redirect, unexpected response"),
                    new OauthException(__($resp->getStatus() . ' :: ' . $resp->getBody()))
                );
            }

            return $resp->getHeader("location");
        } catch (LocalizedException | OauthException $exception) {
            throw $exception;
        } catch (\Exception $exception) {
            $this->logger->critical($exception . '\n' . $exception->getTraceAsString());
            throw new OauthException(
                __('The attempt to post data to consumer failed due to an unexpected error. Please try again later.')
            );
        }
    }

    private function savePbfInstallationHmac($hmac, $scope, $scopeId)
    {
        $this->storeConfigProvider->savePbfInstallHmac($hmac, $scope, $scopeId);
    }

}
