<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Api;

use Magento\Sales\Api\OrderStatusHistoryRepositoryInterface;
use Magento\Sales\Model\Order\Status\HistoryFactory;
use Novuna\Pbf\Api\OrderManagementInterface;
use Magento\Framework\Event\ManagerInterface as EventManagerInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Novuna\Pbf\Model\Sales\Service\FraudCheckService;
use Magento\Sales\Api\OrderAddressRepositoryInterface;
use Magento\Sales\Model\Order\Email\Sender\OrderCommentSender;
use Novuna\Pbf\Model\Webhooks\WebhookSender;

class OrderManagementService implements OrderManagementInterface
{
    private OrderRepositoryInterface $orderRepository;
    private OrderAddressRepositoryInterface $orderAddressRepository;
    private OrderCommentSender $orderCommentSender;
    private EventManagerInterface $eventManager;
    private FraudCheckService $fraudCheckService;
    private OrderStatusHistoryRepositoryInterface $historyRepository;
    private HistoryFactory $historyFactory;
    private WebhookSender $webhookSender;

    public function __construct(
        OrderCommentSender $orderCommentSender,
        OrderAddressRepositoryInterface $orderAddressRepository,
        OrderRepositoryInterface $orderRepository,
        EventManagerInterface $eventManager,
        FraudCheckService $fraudCheckService,
        OrderStatusHistoryRepositoryInterface $historyRepository,
        HistoryFactory $historyFactory,
        WebhookSender $webhookSender
    ) {
        $this->orderCommentSender = $orderCommentSender;
        $this->orderAddressRepository = $orderAddressRepository;
        $this->orderRepository = $orderRepository;
        $this->eventManager = $eventManager;
        $this->fraudCheckService = $fraudCheckService;
        $this->historyRepository = $historyRepository;
        $this->historyFactory = $historyFactory;
        $this->webhookSender = $webhookSender;
    }

    /**
     * @param int $id
     * @param string $type
     * @return \Magento\Sales\Api\Data\OrderAddressInterface|null
     */
    public function getAddress(int $id, string $type = 'billing'): ?\Magento\Sales\Api\Data\OrderAddressInterface
    {
        $order = $this->orderRepository->get($id);
        if ($type == 'billing') {
            $orderAddress = $order->getBillingAddress();
        } else {
            $orderAddress = $order->getShippingAddress();
        }
        return $orderAddress;
    }

    /**
     * @param $id
     * @param \Magento\Sales\Api\Data\OrderAddressInterface $orderAddress
     * @return bool
     */
    public function updateAddresses(
        $id,
        \Magento\Sales\Api\Data\OrderAddressInterface $orderAddress
    ): bool {
        $order = $this->orderRepository->get($id);
        $existingBilling = $order->getBillingAddress();
        $existingBilling->addData($orderAddress->getData());
        //I'd need do both in single transaction ;(
        $this->orderAddressRepository->save($existingBilling);

        $existingShipping = $order->getShippingAddress();
        $existingShipping->addData($orderAddress->getData());
        //I'd need do both in single transaction ;(
        $this->orderAddressRepository->save($existingShipping);

        $this->webhookSender->SendOrderMessage($id, WebhookSender::ORDER_UPDATE, null);
        return true;
    }

    /**
     * Add comment to order
     *
     * @param int $id
     * @param \Magento\Sales\Api\Data\OrderStatusHistoryInterface[] $statusHistoryItems
     * @return bool
     */
    public function addComment($id, $statusHistoryItems)
    {
        $order = $this->orderRepository->get($id);

        foreach ($statusHistoryItems as $statusHistory) {
            $comment = $this->historyFactory->create();
            $comment->setStatus($order->getStatus());
            $comment->setParentId($id);
            $comment->setComment($statusHistory->getComment());
            $comment->setEntityName('order');
            $comment->setIsVisibleOnFront(false);

            if ($statusHistory->getCreatedAt() == null) {
                $createdInLocal = date("Y-m-d H:i:s");
            } else {
                $created = strtotime($statusHistory->getCreatedAt().' UTC');
                $createdInLocal = date("Y-m-d H:i:s", $created);
            }
            $comment->setCreatedAt($createdInLocal);


            $this->historyRepository->save($comment);
        }
        return true;
    }
}
