<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Api;

use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\Sales\Api\OrderRepositoryInterface;
use Novuna\Pbf\Api\Data\DraftOrderDataInterface;
use Novuna\Pbf\Api\Data\OrderMetaFieldDataInterface;
use Novuna\Pbf\Api\Data\OrderMetaFieldWrapperInterface;
use Novuna\Pbf\Api\OrderMetaFieldsInterface;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Model\Config\AppConfig;
use Novuna\Pbf\Model\Data\OrderMetaFieldWrapper;
use Novuna\Pbf\Model\DraftOrder\DraftOrderMaker;
use Novuna\Pbf\Model\DraftOrderRepository;
use Novuna\Pbf\Model\StoreConfigProvider;
use Magento\Quote\Api\CartRepositoryInterface as QuoteRepository;
use Novuna\Pbf\Model\Webhooks\WebhookSender;

class OrderMetaFieldService implements OrderMetaFieldsInterface
{
    private StoreConfigProvider $storeConfigProvider;
    private OrderRepositoryInterface $orderRepository;
    private SalesOrderMetaRepositoryInterface $orderMetaRepository;
    private DraftOrderMaker $draftOrderMaker;
    private QuoteRepository $quoteRepository;
    private WebhookSender $webhookSender;

    public function __construct(
        SalesOrderMetaRepositoryInterface $orderMetaRepository,
        OrderRepositoryInterface $orderRepository,
        DraftOrderMaker $draftOrderMaker,
        QuoteRepository $quoteRepository,
        StoreConfigProvider $storeConfigProvider,
        WebhookSender $webhookSender
    ) {
        $this->orderMetaRepository = $orderMetaRepository;
        $this->draftOrderMaker = $draftOrderMaker;
        $this->orderRepository = $orderRepository;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->quoteRepository = $quoteRepository;
        $this->webhookSender = $webhookSender;
    }

    /**
     * @param int $orderId
     * @param OrderMetaFieldDataInterface $metaField
     * @return OrderMetaFieldWrapperInterface|null
     * @throws WebapiException
     */
    public function createMetaFields(
        int $orderId,
        OrderMetaFieldDataInterface $metaField
    ): ?OrderMetaFieldWrapperInterface {
        try {
            $order = $this->orderRepository->get($orderId);
            $quoteId = (int)$order->getQuoteId();
            $metaField->setOrderId((int)$order->getId());
            $metaField->setQuoteId($quoteId);
            try {
                $saveModel = $this->orderMetaRepository->getByMetaFieldKey(
                    $quoteId,
                    $metaField->getKey(),
                    'quote_id'
                );
                $saveModel->setValue($metaField->getValue());
            } catch (NoSuchEntityException $e) {
                $saveModel = $metaField; //create new meta field if it is not exist
            }
            $this->orderMetaRepository->save($saveModel);
        } catch (NoSuchEntityException $e) {
            throw new WebapiException(__('Order Id is invalid'));
        } catch (LocalizedException $e) {
            throw new WebapiException(__('Unable to save order Meta data: %1', $e->getMessage()));
        }
        return new OrderMetaFieldWrapper($saveModel);
    }

    /**
     * @param string $quoteIdMasked
     * @param OrderMetaFieldDataInterface $metaField
     * @return OrderMetaFieldWrapperInterface|null
     * @throws WebapiException
     */
    public function createDraftOrderMetaFields(
        string $quoteIdMasked,
        OrderMetaFieldDataInterface $metaField
    ): ?OrderMetaFieldWrapperInterface {
        try {
            $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);

            if (!$draftOrder->getQuoteId()) {
                throw new WebapiException(__('Unable to Find Draft order for Quote %1', $quoteIdMasked));
            }
            $metaField->setOrderId($draftOrder->getOrderId());
            $metaField->setQuoteId($draftOrder->getQuoteId());

            $saveModel = $this->orderMetaRepository->getByMetaFieldKey(
                (int)$draftOrder->getQuoteId(),
                $metaField->getKey(),
                'quote_id'
            );
            $saveModel->setValue($metaField->getValue());
        } catch (NoSuchEntityException $e) {
            $saveModel = $metaField; //create new meta field if it is not exist
        }

        try {
            $this->orderMetaRepository->save($saveModel);
        } catch (LocalizedException $e) {
            throw new WebapiException(__('Unable to save order Meta data: %1', $e->getMessage()));
        }
        return new OrderMetaFieldWrapper($saveModel);
    }

    /**
     * @param string $quoteIdMasked
     * @param \Novuna\Pbf\Api\Data\SetOrderFinanceStateInputInterface $input
     *
     * @return bool not used, M2 requires some return type otherwise it crashes
     * @throws WebapiException
     */
    public function setDraftOrderFinanceState(
        string $quoteIdMasked,
        \Novuna\Pbf\Api\Data\SetOrderFinanceStateInputInterface $input
    ):bool
    {
            $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);

            if (!$draftOrder->getQuoteId()) {
                throw new WebapiException(__('Unable to Find Draft order for Quote %1', $quoteIdMasked));
            }
            $metaField = $input->getMetafield();

            $metaField->setOrderId($draftOrder->getOrderId());
            $metaField->setQuoteId($draftOrder->getQuoteId());

            $saveModel = $this->orderMetaRepository->getMetFieldsCollection(
                (int)$draftOrder->getQuoteId(),
                $metaField->getKey(),
                'quote_id')->getFirstItem();

            if (!$saveModel->getId()) {
                $saveModel = $metaField; //create new meta field if it does not exist
            }

            $saveModel->setValue($metaField->getValue());

            $quote = $this->quoteRepository->get($draftOrder->getQuoteId());
            if ($quote->getId() != $draftOrder->getQuoteId()) {
                throw new WebapiException(__('Unable to Find quote for Draft order %1', $draftOrder->getQuoteId()));
            }

            $quote->setData(DraftOrderDataInterface::APPLICATION_NUMBER_FIELD, $input->getAppnumber());
            $quote->setData(DraftOrderDataInterface::FINANCE_STATE_RO_FIELD, $input->getFinancestatestring());


        try {
            $this->quoteRepository->save($quote);
            $this->orderMetaRepository->save($saveModel);
        } catch (LocalizedException $e) {
            throw new WebapiException(__('Unable to save order Meta data: %1', $e->getMessage()));
        }

        return false;
    }

    /**
     * @param int $orderId
     * @param \Novuna\Pbf\Api\Data\SetOrderFinanceStateInputInterface $input
     *
     * @return bool not used, M2 requires some return type otherwise it crashes
     * @throws WebapiException
     */
    public function setOrderFinanceState(int $orderId, \Novuna\Pbf\Api\Data\SetOrderFinanceStateInputInterface $input):bool
    {
        try {
            $order = $this->orderRepository->get($orderId);
            $quoteId = (int)$order->getQuoteId();
            $metaField = $input->getMetafield();

            $metaField->setOrderId($orderId);
            $metaField->setQuoteId($quoteId);

            $order->setData(DraftOrderDataInterface::APPLICATION_NUMBER_FIELD, $input->getAppnumber());
            $order->setData(DraftOrderDataInterface::FINANCE_STATE_RO_FIELD, $input->getFinancestatestring());

            $saveModel = $this->orderMetaRepository->getMetFieldsCollection(
                $quoteId,
                $metaField->getKey(),
                'quote_id')->getFirstItem();

            if (!$saveModel->getId()) {
                $saveModel = $metaField; //create new meta field if it does not exist
            }
            $saveModel->setValue($metaField->getValue());

            $this->orderRepository->save($order);
            $this->orderMetaRepository->save($saveModel);

            $this->webhookSender->SendOrderMessage($orderId, WebhookSender::ORDER_UPDATE, null);

        } catch (NoSuchEntityException $e) {
            throw new WebapiException(__('Order Id is invalid'));
        } catch (LocalizedException $e) {
            throw new WebapiException(__('Unable to save order Meta data: %1', $e->getMessage()));
        }

        return false;
    }


    /**
     * @param int $orderId
     * @param string|null $key
     * @return OrderMetaFieldWrapperInterface|null
     * @throws WebapiException
     */
    public function getMetaFields(
        int $orderId,
        string $key = null
    ): ?OrderMetaFieldWrapperInterface {

        $key = $key ? trim(strip_tags($key)) : null;
        $metaFields = $this->orderMetaRepository->getMetFieldsCollection($orderId, $key)->getItems();

        $metaFieldWrapper = new OrderMetaFieldWrapper();
        $metaFieldWrapper->setMetaFields($metaFields);
        return $metaFieldWrapper;
    }

    /**
     * @param string $quoteIdMasked
     * @param string|null $key
     * @return OrderMetaFieldWrapperInterface|null
     * @throws WebapiException
     */
    public function getDraftOrderMetaFields(
        string $quoteIdMasked,
        string $key = null
    ): ?OrderMetaFieldWrapperInterface {
        $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);

        if (!$draftOrder->getQuoteId()) {
            throw new WebapiException(__('Draft order not available'));
        }

        $key = $key ? trim(strip_tags($key)) : null;
        $metaFields = $this->orderMetaRepository->getMetFieldsCollection(
            (int)$draftOrder->getQuoteId(),
            $key,
            'draft'
        )->getItems();

        $metaFieldWrapper = new OrderMetaFieldWrapper();
        $metaFieldWrapper->setMetaFields($metaFields);
        return $metaFieldWrapper;
    }

    /**
     * @param int $metaFieldId
     * @return bool
     * @throws WebapiException
     */
    public function deleteMetaFields(int $metaFieldId): bool
    {
        try {
            return $this->orderMetaRepository->deleteById($metaFieldId);
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * @param int $metaFieldId
     * @return bool
     * @throws WebapiException
     */
    public function deleteDraftOrderMetaFields(int $metaFieldId): bool
    {
        return $this->deleteMetaFields($metaFieldId);
    }
}
