<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Api;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Api\Data\ScriptTagDataInterface;
use Novuna\Pbf\Api\Data\ScriptTagWrapperInterface;
use Novuna\Pbf\Api\ScriptTagInterface;
use Novuna\Pbf\Model\Data\ScriptTagData;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Data\ScriptTagWrapper;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Magento\Framework\App\RequestInterface;

class ScriptTagService implements ScriptTagInterface
{
    private StoreConfigProvider $storeConfigProvider;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private StoreManagerInterface $storeManager;
    private RequestInterface $request;

    public function __construct(
        StoreConfigProvider $storeConfigProvider,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        StoreManagerInterface $storeManager,
        RequestInterface $request
    ) {
        $this->storeConfigProvider = $storeConfigProvider;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->storeManager = $storeManager;
        $this->request = $request;
    }

    /**
     * POST request to create PBF widget script tag
     * @param ScriptTagDataInterface $scriptTag Script tag input
     * @return ScriptTagWrapperInterface|null
     * @throws \Magento\Framework\Webapi\Exception
     * @throws \Magento\Framework\Exception\StateException
     */
    public function createScriptTag(ScriptTagDataInterface $scriptTag): ?ScriptTagWrapperInterface
    {
        $scopeType = ScopeInterface::SCOPE_STORE;
        //no idea how to get that scope type other than checking the 'all':
        if (strpos($this->request->getRequestUri(), '/rest/all/') === 0) {
            $scopeType = ScopeConfigInterface::SCOPE_TYPE_DEFAULT;
        }

        $storeId = $this->storeManager->getStore()->getId();

        $scriptSrc = trim(strip_tags($scriptTag->getSrc()));
        $scriptTagEvent = str_replace(' ', '_', trim(strip_tags($scriptTag->getEvent())));
        if (!$scriptSrc) {
            throw new WebapiException(__('Script Tag missing'));
        }

        $configId = $this->storeConfigProvider->saveConfig(
            StoreConfigProvider::XML_PATH_PBF_WIDGET_SCRIPT_SRC . '_' . $scriptTagEvent,
            $scriptSrc, $scopeType, $storeId
        );
        $scriptTag = $this->storeConfigProvider->getPbfScriptTagById($configId);
        $scriptTagWrapper = new ScriptTagWrapper();
        $scriptTagWrapper->setScriptTag($this->generateScriptTagData($scriptTag));
        return $scriptTagWrapper;
    }

    /**
     * Query to get PBF widget script tag
     * @return ScriptTagDataInterface[]|null
     * @throws \Magento\Framework\Webapi\Exception
     * @throws \Magento\Framework\Exception\StateException
     */
    public function generateScriptTag(): ?array
    {
        $storeId = $this->storeManager->getStore()->getId();
        $scriptTags = $this->storeConfigProvider->getPbfScriptTags(ScopeInterface::SCOPE_STORE, $storeId);
        $scriptTagsData = [];
        foreach ($scriptTags as $tag) {
            $scriptTagsData[] = $this->generateScriptTagData($tag);
        }

        return $scriptTagsData;
    }

    /**
     * @param $tag
     * @return ScriptTagData
     */
    private function generateScriptTagData($tag): ScriptTagData
    {
        $scriptTagData = new ScriptTagData();
        $scriptTagData->setSrc($tag['value']);
        $scriptTagData->setId($tag['config_id']);
        $scriptTagData->setCreatedAt($tag['updated_at']);
        $eventName = str_replace(
            StoreConfigProvider::XML_PATH_PBF_WIDGET_SCRIPT_SRC . '_',
            '',
            $tag['path']
        );
        $scriptTagData->setEvent($eventName);
        return $scriptTagData;
    }

    /**
     * @return ScriptTagWrapperInterface|null
     * @throws WebapiException
     * @throws \Magento\Framework\Exception\StateException
     */
    public function getScriptTag(): ?ScriptTagWrapperInterface
    {
        return new ScriptTagWrapper($this->generateScriptTag());
    }

    /**
     * @param int $scriptTagId
     * @return bool
     * @throws WebapiException
     */
    public function deleteScriptTag($scriptTagId): bool
    {
        $this->storeConfigProvider->deleteConfigById($scriptTagId);
        return true;
    }
}
