<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Novuna\Pbf\Api\Data\OrderMetaFieldDataInterface;
use Novuna\Pbf\Api\Data\OrderRiskDataInterface;
use Novuna\Pbf\Api\OrderRiskRepositoryInterface;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Model\Data\OrderMeta;
use Novuna\Pbf\Model\Data\OrderRiskData;
use Novuna\Pbf\Model\ResourceModel\OrderRisk\Collection;
use Novuna\Pbf\Model\ResourceModel\OrderRisk\CollectionFactory as OrderRiskCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class OrderRiskRepository implements OrderRiskRepositoryInterface
{
    /**
     * @var \Novuna\Pbf\Model\ResourceModel\OrderRisk
     */
    protected \Novuna\Pbf\Model\ResourceModel\OrderRisk $resource;

    /**
     * @var OrderRiskFactory
     */
    protected OrderRiskFactory $orderRiskFactory;

    /**
     * @var OrderRiskCollectionFactory
     */
    protected OrderRiskCollectionFactory $orderRiskCollectionFactory;

    private ?CollectionProcessorInterface $collectionProcessor;

    private SearchResultsInterface $searchResultsFactory;


    public function __construct(
        \Novuna\Pbf\Model\ResourceModel\OrderRisk $resource,
        OrderRiskFactory $orderRiskFactory,
        OrderRiskCollectionFactory $orderRiskCollectionFactory,
        SearchResultsInterface $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->orderRiskFactory = $orderRiskFactory;
        $this->orderRiskCollectionFactory = $orderRiskCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * Save order risk.
     *
     * @param \Novuna\Pbf\Api\Data\OrderRiskDataInterface $risk
     * @return \Novuna\Pbf\Api\Data\OrderRiskDataInterface
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function save(\Novuna\Pbf\Api\Data\OrderRiskDataInterface $risk)
    {
        try {
            $this->resource->save($risk);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $risk;
    }

    /**
     * Retrieve order risks.
     *
     * @param int $orderRiskId
     * @return \Novuna\Pbf\Api\Data\OrderRiskDataInterface
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getById($orderRiskId)
    {
        $orderRisk = $this->orderRiskFactory->create();
        $this->resource->load($orderRisk, $orderRiskId);
        if (!$orderRisk->getId()) {
            throw new NoSuchEntityException(
                __('The order risk with the "%1" ID doesn\'t exist.', $orderRiskId)
            );
        }

        return $orderRisk;
    }

    public function getOrderRisks(
        int $orderId): Collection
    {
        $collection = $this->orderRiskCollectionFactory->create()
            ->addFieldToFilter(OrderRiskData::ORDER_ID, $orderId);

        return $collection;
    }

    public function existRisk(
        int $orderId, \Novuna\Pbf\Api\Data\OrderRiskDataInterface $risk
    ): ?OrderRiskDataInterface
    {
        $collection = $this->orderRiskCollectionFactory->create()
            ->addFieldToFilter(OrderRiskData::ORDER_ID, $orderId)
            ->addFieldToFilter(OrderRiskData::MESSAGE, $risk->getMessage())
            ->addFieldToFilter(OrderRiskData::SCORE, $risk->getScore())
            ->addFieldToFilter(OrderRiskData::RECOMMENDATION, $risk->getRecommendation())
        ;

        if ($collection->count() == 0) {
            return null;
        }
        
        return $collection->getFirstItem();
    }


    /**
     * Delete SalesOrderMeta by given Identity
     *
     * @param string $riskId
     *
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById($riskId)
    {
        $this->resource->delete($this->getById($riskId));
    }

    public function deleteAll($orderId)
    {
        $this->resource->getConnection()->delete(
            $this->resource->getMainTable(),
            [ OrderRiskData::ORDER_ID . ' = ?' => $orderId]
        );
    }
}
