<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\ResourceModel;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;

class StoreConfig extends \Magento\Config\Model\ResourceModel\Config
{
    /**
     * @param $path
     * @param string $scope
     * @param int $scopeId
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getConfig($path, $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0, $fetchAll = false)
    {
        //Fix discrepancy between admin FE and database:
        if ($scope == ScopeInterface::SCOPE_WEBSITE) {
            $scope = ScopeInterface::SCOPE_WEBSITES;
        }

        if ($scope == ScopeInterface::SCOPE_STORE) {
            $scope = ScopeInterface::SCOPE_STORES;
        }
        
        $connection = $this->getConnection();
        $select = $connection->select()->from(
            $this->getMainTable()
        )->where(
            'path LIKE ?',
            $path . '%'
        )->where(
            'scope = ?',
            $scope
        )->where(
            'scope_id = ?',
            $scopeId
        );
        return $fetchAll ? $connection->fetchAll($select) : $connection->fetchRow($select);
    }

    /**
     * @param $configId
     * @return mixed
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function getConfigById($configId)
    {
        $connection = $this->getConnection();
        $select = $connection->select()->from(
            $this->getMainTable()
        )->where(
            $this->getIdFieldName() . ' = ?',
            $configId
        );
        return $connection->fetchRow($select);
    }

    /**
     * Save config value
     *
     * @param string $path
     * @param string $value
     * @param string $scope
     * @param int $scopeId
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function saveConfig($path, $value, $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0)
    {
        //Fix discrepancy between admin FE and database:
        if ($scope == ScopeInterface::SCOPE_WEBSITE) {
            $scope = ScopeInterface::SCOPE_WEBSITES;
        }

        if ($scope == ScopeInterface::SCOPE_STORE) {
            $scope = ScopeInterface::SCOPE_STORES;
        }

        $connection = $this->getConnection();
        $row = $this->getConfig($path, $scope, $scopeId);

        $newData = ['scope' => $scope, 'scope_id' => $scopeId, 'path' => $path, 'value' => $value];

        if ($row) {
            $whereCondition = [$this->getIdFieldName() . '=?' => $row[$this->getIdFieldName()]];
            $connection->update($this->getMainTable(), $newData, $whereCondition);
            $configId = $row[$this->getIdFieldName()];
        } else {
            $connection->insert($this->getMainTable(), $newData);
            $configId = $connection->lastInsertId();
        }
        return $configId;
    }

    /**
     * Delete Config By ID
     * @param $configId
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function deleteConfigById($configId)
    {
        $connection = $this->getConnection();
        $connection->delete(
            $this->getMainTable(),
            [$this->getIdFieldName() . '=?' => $configId]
        );
        return $this;
    }

    /**
     * Delete Config By paths
     * @param array $paths
     * @return $this
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    public function deleteConfigByPaths(array $paths = [], $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0): StoreConfig
    {
        //Fix discrepancy between admin FE and database:
        if ($scope == ScopeInterface::SCOPE_WEBSITE) {
            $scope = ScopeInterface::SCOPE_WEBSITES;
        }

        if ($scope == ScopeInterface::SCOPE_STORE) {
            $scope = ScopeInterface::SCOPE_STORES;
        }

        $connection = $this->getConnection();
        $connection->delete(
            $this->getMainTable(),
            ['path IN(?)' => $paths, 'scope' => $scope, 'scope_id' => $scopeId]
        );
        return $this;
    }
}
