<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Magento\Framework\Api\SearchCriteriaInterface;
use Magento\Framework\Api\SearchResultsInterface;
use Novuna\Pbf\Api\Data\OrderMetaFieldDataInterface;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Model\Data\OrderMeta;
use Novuna\Pbf\Model\ResourceModel\SalesOrderMeta as ResourceSalesOrderMeta;
use Novuna\Pbf\Model\ResourceModel\SalesOrderMeta\Collection;
use Novuna\Pbf\Model\ResourceModel\SalesOrderMeta\CollectionFactory as SalesOrderMetaCollectionFactory;
use Magento\Framework\Api\SearchCriteria\CollectionProcessorInterface;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class SalesOrderMetaRepository implements SalesOrderMetaRepositoryInterface
{
    /**
     * @var ResourceSalesOrderMeta
     */
    protected ResourceSalesOrderMeta $resource;

    /**
     * @var SalesOrderMetaFactory
     */
    protected SalesOrderMetaFactory $salesOrderMetaFactory;

    /**
     * @var SalesOrderMetaCollectionFactory
     */
    protected SalesOrderMetaCollectionFactory $salesOrderMetaCollectionFactory;

    private ?CollectionProcessorInterface $collectionProcessor;

    private SearchResultsInterface $searchResultsFactory;

    /**
     * @param ResourceSalesOrderMeta $resource
     * @param SalesOrderMetaFactory $salesOrderMetaFactory
     * @param SalesOrderMetaCollectionFactory $salesOrderMetaCollectionFactory
     * @param SearchResultsInterface $searchResultsFactory
     * @param CollectionProcessorInterface|null $collectionProcessor
     */
    public function __construct(
        ResourceSalesOrderMeta $resource,
        SalesOrderMetaFactory $salesOrderMetaFactory,
        SalesOrderMetaCollectionFactory $salesOrderMetaCollectionFactory,
        SearchResultsInterface $searchResultsFactory,
        CollectionProcessorInterface $collectionProcessor
    ) {
        $this->resource = $resource;
        $this->salesOrderMetaFactory = $salesOrderMetaFactory;
        $this->salesOrderMetaCollectionFactory = $salesOrderMetaCollectionFactory;
        $this->searchResultsFactory = $searchResultsFactory;
        $this->collectionProcessor = $collectionProcessor;
    }

    /**
     * Save SalesOrderMeta data
     *
     * @param OrderMetaFieldDataInterface $orderMetaField
     *
     * @return OrderMetaFieldDataInterface $orderMetaField
     * @throws CouldNotSaveException
     */
    public function save(OrderMetaFieldDataInterface $orderMetaField)
    {
        try {
            $this->resource->save($orderMetaField);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $orderMetaField;
    }

    /**
     * Load SalesOrderMeta data by given SalesOrderMeta Identity
     *
     * @param string $metaFieldId
     *
     * @return SalesOrderMeta
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function getById($metaFieldId)
    {
        $salesOrderMeta = $this->salesOrderMetaFactory->create();
        $this->resource->load($salesOrderMeta, $metaFieldId);
        if (!$salesOrderMeta->getId()) {
            throw new NoSuchEntityException(
                __('The sales Order Meta with the "%1" ID doesn\'t exist.', $metaFieldId)
            );
        }

        return $salesOrderMeta;
    }

    /**
     * Load SalesOrderMeta data by given SalesOrderMeta Identity
     *
     * @param int $orderId
     * @param string $metaFieldKey
     * @param string $type
     * @return OrderMetaFieldDataInterface
     * @throws NoSuchEntityException
     */
    public function getByMetaFieldKey(
        int $orderId,
        string $metaFieldKey,
        string $type = 'order'
    ): OrderMetaFieldDataInterface {
        $collection = $this->getMetFieldsCollection($orderId, $metaFieldKey, $type);
        $salesOrderMeta = $collection->getFirstItem();

        if (!$salesOrderMeta->getId()) {
            throw new NoSuchEntityException(
                __('The sales Order Meta with the "%1" Key doesn\'t exist.', $metaFieldKey)
            );
        }

        return $salesOrderMeta;
    }

    /**
     * Retrieve orderMetaField.
     *
     * @param int $orderId
     * @param string|null $metaFieldKey
     * @param string $type
     * @return Collection
     */
    public function getMetFieldsCollection(
        int $orderId,
        string $metaFieldKey = null,
        string $type = 'order'
    ): Collection {
        $filterField = ($type == 'order') ? OrderMeta::META_DATA_ORDER_ID : OrderMeta::META_DATA_QUOTE_ID;
        $collection = $this->getCollection()
            ->addFieldToFilter($filterField, $orderId);
        if ($metaFieldKey) {
            $collection->addFieldToFilter(OrderMeta::META_DATA_KEY, $metaFieldKey);
        }

        return $collection;
    }

    /**
     * @return Collection
     */
    public function getCollection()
    {
        return $this->salesOrderMetaCollectionFactory->create();
    }

    /**
     * Load SalesOrderMeta data collection by given search criteria
     *
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @param SearchCriteriaInterface $searchCriteria
     * @return \Magento\Framework\Api\SearchResultsInterface
     */
    public function getList(SearchCriteriaInterface $searchCriteria): SearchResultsInterface
    {

        $collection = $this->getCollection();
        $this->collectionProcessor->process($searchCriteria, $collection);

        $searchResults = $this->searchResultsFactory->create();
        $searchResults->setSearchCriteria($searchCriteria);
        $searchResults->setItems($collection->getItems());
        $searchResults->setTotalCount($collection->getSize());

        return $searchResults;
    }

    /**
     * Delete SalesOrderMeta
     *
     * @param OrderMetaFieldDataInterface $orderMetaField
     *
     * @return bool
     * @throws CouldNotDeleteException
     */
    public function delete(OrderMetaFieldDataInterface $orderMetaField): bool
    {
        try {
            $this->resource->delete($orderMetaField);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * Delete SalesOrderMeta by given Identity
     *
     * @param string $metaFieldId
     *
     * @return bool
     * @throws CouldNotDeleteException
     * @throws NoSuchEntityException
     */
    public function deleteById($metaFieldId): bool
    {
        return $this->delete($this->getById($metaFieldId));
    }
}
