<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Total\Invoice;

use Magento\Sales\Model\Order\Invoice;
use Magento\Sales\Model\Order\Invoice\Total\AbstractTotal;
use Novuna\Pbf\Model\Utils\TotalCalculateUtils;

class FinanceAmount extends AbstractTotal
{

    /**
     * Add financed amount to totals so grandTotal ( = sum that will should captured) will reflect the total
     * - financed amount (user will pay only deposit)
     *
     * @param Invoice $invoice
     * @return $this
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function collect(Invoice $invoice)
    {
        $order = $invoice->getOrder();
        if ($order->getFinanceAmount() &&
            !TotalCalculateUtils::bcCompare(
                floatval($order->getFinanceAmountInvoiced()),
                floatval($order->getFinanceAmount())
            )
        ) {
            $amount = abs(floatval($order->getFinanceAmount()));
            $baseAmount = abs(floatval($order->getBaseFinanceAmount()));

            if (TotalCalculateUtils::bcCompare($amount, 0)) {
                return $this;
            }

            $invoice->setFinanceAmount(-$amount);
            $invoice->setBaseFinanceAmount(-$baseAmount);

            $grandTotal = abs(floatval($invoice->getGrandTotal() - $amount)) < 0.0001
                ? 0 : $invoice->getGrandTotal() - $amount;
            $baseGrandTotal = abs(floatval($invoice->getBaseGrandTotal() - $baseAmount)) < 0.0001
                ? 0 : $invoice->getBaseGrandTotal() - $baseAmount;
            $invoice->setGrandTotal($grandTotal);
            $invoice->setBaseGrandTotal($baseGrandTotal);
        }

        return $this;
    }
}
