<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Model;

use Novuna\Pbf\Api\Data\WebhookDataInterface;
use Novuna\Pbf\Api\WebhookRepositoryInterface;
use Novuna\Pbf\Model\ResourceModel\Webhook as ResourceWebhook;
use Novuna\Pbf\Model\ResourceModel\Webhook\Collection;
use Novuna\Pbf\Model\ResourceModel\Webhook\CollectionFactory as WebhookCollectionFactory;
use Magento\Framework\Exception\CouldNotDeleteException;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class WebhookRepository implements WebhookRepositoryInterface
{
    protected ResourceWebhook $resource;
    protected WebhookFactory $webhookFactory;
    protected WebhookCollectionFactory $webhookCollectionFactory;

    public function __construct(
        ResourceWebhook $resource,
        WebhookFactory $webhookFactory,
        WebhookCollectionFactory $webhookCollectionFactory
    ) {
        $this->resource = $resource;
        $this->webhookFactory = $webhookFactory;
        $this->webhookCollectionFactory = $webhookCollectionFactory;
    }

    /**
     * @inheritDoc
     */
    public function save(WebhookDataInterface $webhook): WebhookDataInterface
    {
        try {
            $this->resource->save($webhook);
        } catch (\Exception $exception) {
            throw new CouldNotSaveException(__($exception->getMessage()));
        }

        return $webhook;
    }

    /**
     * @inheritDoc
     */
    public function getById(int $wehookId): WebhookDataInterface
    {
        $webhook = $this->webhookFactory->create();
        $this->resource->load($webhook, $wehookId);
        if (!$webhook->getId()) {
            throw new NoSuchEntityException(
                __('The webhook with the "%1" ID doesn\'t exist.', $wehookId)
            );
        }

        return $webhook;
    }

    public function getByTopic(string $topic): Collection
    {
        $webhookCollection = $this->webhookCollectionFactory->create();
        $webhookCollection->addFieldToFilter('topic', ['like' => "%$topic%"]);

        return $webhookCollection;
    }

    /**
     * @inheritDoc
     */
    public function getCollection(): Collection
    {
        return $this->webhookCollectionFactory->create();
    }

    /**
     * @inheritDoc
     */
    public function delete(WebhookDataInterface $webhook): bool
    {
        try {
            $this->resource->delete($webhook);
        } catch (\Exception $exception) {
            throw new CouldNotDeleteException(__($exception->getMessage()));
        }

        return true;
    }

    /**
     * @inheritDoc
     */
    public function deleteById(int $webhook): bool
    {
        return $this->delete($this->getById($webhook));
    }
}
