<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Model\Webhooks;

use Magento\Sales\Model\Order;
use Magento\Framework\Event\Observer;
use Magento\Framework\HTTP\Client\Curl;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Api\Data\WebhookDataInterface;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\HTTP\Client\CurlFactory;
use Novuna\Pbf\Api\WebhookRepositoryInterface;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Magento\Framework\MessageQueue\PublisherInterface;

/**
 * This class is meant to be the utility for sending webhooks to registered webhooks endpoints.
 * It uses Message Queue under the hood for retries of unsuccessfull trial.
 */
class WebhookSender
{
    //Topic constants that are common with appserver. If changed, change it on webhooks installation on appserver.
    const ORDER_UPDATE = 'order_update';
    const ORDER_SHIPPED = 'order_shipped';
    const ORDER_CANCEL = 'order_cancel';
    const DRAFT_CONVERTED = 'draft_converted';

    private PbfLoggerInterface $logger;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private WebhookRepositoryInterface $webhookRepository;
    private CurlFactory $curlFactory;
    private PublisherInterface $publisher;

    public function __construct(
        PbfLoggerInterface $logger,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        WebhookRepositoryInterface $webhookRepository,
        CurlFactory $curlFactory,
        PublisherInterface $publisher
    ) {
        $this->publisher = $publisher;
        $this->logger = $logger;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->webhookRepository = $webhookRepository;
        $this->curlFactory = $curlFactory;
    }

    public function SendOrderMessage($orderId, string $topic, $data)
    {
        //Prevent signing "nothing" :
        if ($data == null) {
            $data = ["nonce" =>bin2hex(random_bytes(32))];
        }

        $webhooks = $this->webhookRepository->getByTopic($topic);
        /** @var WebhookDataInterface $webhook */
        foreach ($webhooks as $webhook) {
            $url = $webhook->getDeliveryUrl();
            $curl = $this->curlFactory->create();
            $headers = [
                'content-type' => 'application/json',
                'X-Webhook-Signature' => $this->makeSignature($data, $webhook->getSecret()),
                'X-Webhook-Shop-Domain' => $this->pbfWidgetConfigProvider->getShopDomain(false), // M2 lost context of the store, if it was `default` or default one referred by storecode
                'X-Webhook-Event' => $webhook->getTopic(),
                'X-Webhook-Resource-Id' => $orderId,
            ];
            $curl->setHeaders($headers);
            //When statusCode > 400 => throw exception:
            $curl->setOption(CURLOPT_FAILONERROR, 1); // difficult to combine return-error and throw-exception approaches, unify it to throw only

            try {
                $curl->post($url, json_encode($data));
            } catch (\Exception $e) {
                $this->logger->error("error when sending webhook message: " . $e->getMessage());
                try {
                    $this->scheduleWebhook($url, $headers, $data);
                    $this->logger->info("webhook was added to queue for retry, original error:" . $e->getMessage());
                } catch (\Exception $e) {
                    $this->logger->error("error when scheduling webhook: " . $e->getMessage());
                }
            }
        }
    }

    private function makeSignature($payload, string $secret): string
    {
        if (is_array($payload)) {
            $payload = json_encode($payload);
        }
        return base64_encode(hash_hmac('sha256', (string)$payload, $secret, true));
    }

    private function scheduleWebhook($url, $headers, $data) {
        $publishData = [
            'url' => $url,
            'headers' => $headers,
            'data' => $data
        ];

        $this->publisher->publish('pbf_order.updated', json_encode($publishData));
    }
}
