<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Observer\Checkout;

use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Quote\Model\Quote;
use Magento\Quote\Api\CartRepositoryInterface as CartRepository;
use Novuna\Pbf\Model\Config\AppConfig;
use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\UrlInterface;
use Magento\Framework\App\Response\Http as MagentoResponse;

class QuoteRedirect implements ObserverInterface
{
    private CartRepository $cartRepository;
    private CheckoutSession $checkoutSession;
    private QuoteIdManager $quoteIdManager;
    private CustomerSession $customerSession;
    private UrlInterface $urlInterface;
    private MagentoResponse $response;

    public function __construct(
        CheckoutSession $checkoutSession,
        QuoteIdManager $quoteIdManager,
        CartRepository $cartRepository,
        CustomerSession $customerSession,
        UrlInterface $urlInterface,
        MagentoResponse $response
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->customerSession = $customerSession;
        $this->quoteIdManager = $quoteIdManager;
        $this->response = $response;
        $this->cartRepository = $cartRepository;
        $this->urlInterface = $urlInterface;
    }

    public function execute(Observer $observer)
    {
        $request = $observer->getEvent()->getRequest();

        // If no param is set then we always want to double check it isnt a locked quote
        // and to reset if it is
        $quoteId = $request->getParam(AppConfig::INBOUND_URL_PARAMETER);

        if (!$quoteId) {
            return;
        }

        $unmaskedQuoteId = null;
        if ($this->quoteIdManager->isIdMasked($quoteId)) {
            $unmaskedQuoteId = (int)$this->quoteIdManager->getUnMaskedById($quoteId);
        }

        if ($unmaskedQuoteId) {
            $paramQuote = $this->cartRepository->get($unmaskedQuoteId);

            if ($this->isQuoteValid($paramQuote)) {
                if (!$paramQuote->getCustomerIsGuest() && !$this->customerSession->isLoggedIn()) {
                    // redirect to login if quote is from a non-guest account, otherwise it'll not apply and lead to cart instead
                    // we add fragment here for final checkout step as fragments are strictly clientside and we don't see them serverside
                    $this->customerSession->setAfterAuthUrl($this->urlInterface->getCurrentUrl() . "#payment");
                    $this->customerSession->authenticate();
                    // response is shared, authenticate waits for magento to resolve the response but we have to send it now
                    $this->response->send();
                    // end here, otherwise magento will override redirect location and other data, not elegant but least roundabout way
                    die;
                }
            }
        }
    }

    private function isQuoteValid(Quote $quote): bool
    {
        if (!$quote->getData(AppConfig::DRAFT_ORDER_FLAG)) {
            return false;
        }

        if ($quote->getData(AppConfig::DRAFT_ORDER_CONVERTED_FLAG)) {
            return false;
        }

        return true;
    }
}
