<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Plugin\DraftOrder\Model;

use Novuna\Pbf\Model\DraftOrder\Quote\QuoteLocker;
use Magento\Framework\Exception\LocalizedException;
use Magento\Checkout\Model\Cart;

/**
 * Collection of plugins being used to block updates to the cart due
 * to the current quote being considered a draft order.
 */
class CartBlocker
{
    private QuoteLocker $quoteLocker;

    public function __construct(QuoteLocker $quoteLocker)
    {
        $this->quoteLocker = $quoteLocker;
    }

    /**
     * Prevent updating cart items the quote is considered locked due to it being
     * a draft order.
     */
    public function beforeUpdateItem(Cart $cart)
    {
        $this->blockAction($cart);
    }

    /**
     * Prevent updating cart items the quote is considered locked due to it being
     * a draft order.
     */
    public function beforeUpdateItems(Cart $cart)
    {
        $this->blockAction($cart);
    }

    /**
     * Prevent removing from cart if the quote is considered locked due to it being
     * a draft order.
     */
    public function beforeRemoveItem(Cart $cart)
    {
        $this->blockAction($cart);
    }

    /**
     * Prevent adding to cart if the quote is considered locked due to it being
     * a draft order.
     */
    public function beforeAddProduct(Cart $cart)
    {
        $this->blockAction($cart);
    }

    private function blockAction(Cart $cart)
    {
        if ($this->quoteLocker->isQuoteLocked($cart->getQuote())) {
            throw new LocalizedException(__('Could not update cart item'));
        }
    }
}
