<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\ViewModel\Checkout\Success;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\DataObjectFactory;
use Magento\Framework\Event\ManagerInterface as EventManager;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\Serialize\SerializerInterface;
use Magento\Framework\UrlInterface;
use Magento\Quote\Model\Quote;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteIdManager;
use Novuna\Pbf\Model\PaymentMethod\PaymentMethodAvailability;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;
use Novuna\Pbf\Observer\Checkout\SalesOrderSaveAfter;
use Novuna\Pbf\ViewModel\AbstractPbfWidgetProvider;

class PbfWidget extends \Novuna\Pbf\ViewModel\Checkout\PbfWidget
{
    private SalesOrderMetaRepositoryInterface $orderMetaRepository;
    private PbfLoggerInterface $logger;

    public function __construct(
        EventManager $eventManager,
        DataObjectFactory $dataObjectFactory,
        CheckoutSession $checkoutSession,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        Json $serializer,
        UrlInterface $urlBuilder,
        PbfLoggerInterface $logger,
        SalesOrderMetaRepositoryInterface $orderMetaRepository,
        QuoteIdManager $quoteIdManager,
        PaymentMethodAvailability $paymentMethodAvailability,
        StoreManagerInterface $storeManager
    ) {
        $this->orderMetaRepository = $orderMetaRepository;
        $this->logger = $logger;
        parent::__construct(
            $eventManager,
            $dataObjectFactory,
            $checkoutSession,
            $pbfWidgetConfigProvider,
            $serializer,
            $urlBuilder,
            $quoteIdManager,
            $paymentMethodAvailability,
            $storeManager
        );
    }

    public function getScriptSrc($eventName = ''): ?string
    {
        $scriptSrc = $this->pbfWidgetConfigProvider->getScriptSrc($eventName);
        return $scriptSrc != '' ? $scriptSrc :
            $this->pbfWidgetConfigProvider->getAppServerUrl() . PbfWidgetConfigProvider::THANK_YOU_JS_PATH;
    }

    public function getOrderConfig(): string
    {
        $lastOrder = $this->checkoutSession->getLastRealOrder();
        $orderConfig = [];
        //check order is placed with PBF
        if ($lastOrder && $lastOrder->getFinanceEnabled()) {
            $useDepSecond = $this->getQuote()->getDeposit2nd();
            $lastOrderId = (int)$lastOrder->getId();
            try {
                $orderMetaData = $this->orderMetaRepository->getByMetaFieldKey(
                    $lastOrderId,
                    SalesOrderSaveAfter::ORDER_FINANCE_STATE_META_KEY
                );
                $metaData = $this->serializer->unserialize($orderMetaData->getValue());
                $orderConfig[] = "let order_id =" . $lastOrderId;
                $orderConfig[] = "useDepSecond=" . (int)($useDepSecond ?? 0);
                $orderConfig[] = "let serverVersion = '" . ($metaData['serverVersion'] ?? '1.0.0.0') . "';";

            } catch (NoSuchEntityException $e) {
                $this->logger->error($e->getMessage());
            }
        }
        return implode('; ', $orderConfig);
    }

    public function isLastOrderFinanced(): bool
    {
        $lastOrder = $this->checkoutSession->getLastRealOrder();
        return $lastOrder && $lastOrder->getFinanceEnabled();
    }

    public function isOrderDep2nd(): bool
    {
        $lastOrder = $this->checkoutSession->getLastRealOrder();

        if ($lastOrder && $lastOrder->getFinanceEnabled()) {
            return (bool) $this->getQuote()->getDeposit2nd();
        }

        return false;
    }
}
