/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
define([
    'jquery',
    'ko',
    'underscore',
    'mage/storage',
    'Magento_Checkout/js/model/quote',
    'Magento_Checkout/js/model/cart/totals-processor/default',
    'Magento_Checkout/js/view/billing-address',
    'Magento_Checkout/js/action/get-payment-information',
    'Magento_Checkout/js/action/select-shipping-address',
    'Magento_Checkout/js/action/select-billing-address',
    'Magento_Checkout/js/model/checkout-data-resolver',
    'Magento_Checkout/js/action/set-billing-address',
    'Magento_Ui/js/model/messageList',
    'Magento_Checkout/js/checkout-data',

], function ($, ko, _, storage, quote, totalsDefaultProvider, billingAddress, getPaymentInformationAction, selectShippingAddress, selectBillingAddress, checkoutDataResolver, setBillingAddressAction, globalMessageList, checkoutData) {
    'use strict';

    return function (data) {
        var self = this;
        var isSending = false;
        self.isCheckout = data.isCheckout || false;
        self.financeEnabled = ko.observable(data.financeEnabled);

        //Watching event: adding totals table
        var observer = new MutationObserver(function (mutations) {
            mutations.forEach(function (mutation) {
                var jq = $(mutation.addedNodes);
                if (jq.is("table.data.table.table-totals")) {
                    if (self.financeEnabled()) {
                        self.addTotalsWarning();
                    }
                    observer.disconnect();
                }
            });
        });

        var payments_observer = new MutationObserver(function (mutations) {
            mutations.forEach(function (mutation) {
                var jq = $(mutation.addedNodes);
                if (jq.is("div.payment-method")) {
                    if (self.financeEnabled()) {
                        $("div.payment-group").addClass("pbffinance-on");
                    }
                }
            });
        });

        $(document).on('click', '#pbf-total-collector', function (e) {
            self.loadFromServer();
            totalsDefaultProvider.estimateTotals(quote.shippingAddress());

            if (self.isCheckout &&
                quote.billingAddress() &&
                quote.billingAddress().street &&
                quote.billingAddress().postcode) {
                getPaymentInformationAction($.Deferred());
            }
        });

        $(document).on('click', '.button.action.continue.primary', function () {
            self.triggerAddressUpdate(billingAddress);
        });


        self.loadFromServer = function () {
            var deferred = $.Deferred();
            var serviceUrl = '/pbfwidget/cart';
            var self = this;
            return storage.get(
                serviceUrl, false
            ).done(function (response) {

                if (response.pbfApplied) {
                    self.financeEnabled(true);
                    self.hideNonUKAddresses();
                } else {
                    self.financeEnabled(false);
                    self.unhideNonUKAddresses(false)
                }
                deferred.resolve();
            }).fail(function (response) {
                deferred.reject();
            });
        };
        /**
         * Set initial value for unable change shipping address when financed
         */
        quote.shippingAddress.subscribe(function () {
            if (self.isCheckout) {
                quote.shippingAddress().canUseForBilling(true);
                if (self.financeEnabled()) {
                    self.resetShippingAddress();
                    quote.shippingAddress().canUseForBilling(false);

                    if ($('.shipping-address-item').length > 1) {
                        return;
                    }
                    self.hideNonUKAddresses();
                }
            }
        });

        /**
         * Disabling shipping address when finance is on
         */
        self.financeEnabled.subscribe(function (newValue) {
            if (newValue) {
                if (self.isCheckout) {
                    billingAddress().cancelAddressEdit();
                    billingAddress().isAddressSameAsShipping(false); // false here means billing address is same as shipping

                    if (quote.shippingAddress()) {
                        self.resetShippingAddress();
                        quote.shippingAddress().canUseForBilling(false);
                    }

                    $("div.payment-group").addClass("pbffinance-on");
                }
                self.hideNonUKAddresses();
                self.addTotalsWarning();

            } else {
                if (self.isCheckout) {
                    $("#pbf-address-not-possible-error").remove();
                    $("#pbf-address-not-possible-error-shipping").remove();
                    $("#shipping").removeClass("pbffinance-address-on");

                    if (quote.shippingAddress()) {
                        quote.shippingAddress().canUseForBilling(true);
                    }
                    $("div.payment-group").removeClass("pbffinance-on");
                }
                self.unhideNonUKAddresses();
                self.hideTotalsWarning();
            }
        });

        self.updateAddresses = function () {

            if (window.checkoutConfig.reloadOnBillingAddress ||
                !window.checkoutConfig.displayBillingOnPaymentMethod
            ) {
                setBillingAddressAction(globalMessageList);
            }
        };
        /**
         * Trigger Address update
         */
        self.triggerAddressUpdate = function (billingAddress) {

            if (!self.financeEnabled() || !self.isCheckout) {
                return;
            }

            if (quote.billingAddress() === null) {
                return;
            }

            let billingPostcode = quote.billingAddress().postcode;


            checkoutDataResolver.resolveShippingAddress();
            checkoutDataResolver.resolveBillingAddress();

            if (!billingPostcode) {
                console.log('fix-trigger');
                selectBillingAddress(quote.shippingAddress());
                self.updateAddresses();
                checkoutData.setSelectedBillingAddress(null);

            }
        };

        /**
         * Do not allow a different saved billing address to be used
         */
        self.resetShippingAddress = function () {

            if (!self.isCheckout) {
                return;
            }
            $("#shipping").addClass("pbffinance-address-on");
            if ($("#pbf-address-not-possible-error").length == 0) {
                $(".checkout-payment-method .payment-method._active .payment-method-title.field.choice").prepend('<div id="pbf-address-not-possible-error">It is not possible to select different shipping address when the order is being paid by finance - billing address will be used as shipping</div>');
            }
            if ($("#pbf-address-not-possible-error-shipping").length == 0) {
                $("#shipping .field.addresses").prepend('<div id="pbf-address-not-possible-error-shipping">It is not possible to select different shipping address when the order is being paid by finance - billing address will be used as shipping</div>');
            }
            if (quote.billingAddress() == null) {
                return;
            }

            if (!quote || !quote.shippingAddress() || !quote.billingAddress()) {
                return;
            }

            if (!quote.shippingAddress().customerAddressId || !quote.billingAddress().customerAddressId) {
                return;
            }

            var pbfShippingId = quote.shippingAddress().customerAddressId;
            var pbfBillingId = quote.billingAddress().customerAddressId;

            let billingPostcode = quote.billingAddress().postcode;
            let shippingPostcode = quote.shippingAddress().postcode;

            if (!billingPostcode || !shippingPostcode) {
                return;
            }

            // These checks are to prevent a recurrsive loop
            if (pbfShippingId && pbfBillingId && pbfShippingId != pbfBillingId) {
                selectShippingAddress(quote.billingAddress());
            }
        }

        /**
         * Add totals warning
         */
        self.addTotalsWarning = function () {
            $(".data.table.table-totals").addClass("finance-enabled");
            $(".data.table.totals").addClass("finance-enabled");
        };

        /**
         * Remove totals warning
         */
        self.hideTotalsWarning = function () {
            $(".data.table.table-totals").removeClass("finance-enabled");
            $(".data.table.totals").removeClass("finance-enabled");
        };

        /**
         * Hide/show My billing and shipping address are the same message and checkbox
         */
        self.makeBillingAndShippingAddressSameVisible = function (visible) {
            var sel = $(".billing-address-same-as-shipping-block.field.choice");
            if (visible) {
                sel.show();
            } else {
                sel.hide();
            }
        }

        /**
         * Hide all non UK addresses
         */
        self.hideNonUKAddresses = function () {
            var $countrySelector = $("select[name='country_id']");
            if ($countrySelector.length) {
                $countrySelector.children("[value!='GB']").hide();
                $countrySelector.val('GB').change();
            }
        };

        /**
         * Unhide all non uk addresses
         */
        self.unhideNonUKAddresses = function () {
            var $countrySelector = $("select[name='country_id']");
            if ($countrySelector.length) {
                $countrySelector.children("[value!='GB']").show();
            }
        };

        self.updateCarTotalsJson = function (totals, isFinanceEnabled) {
            let destElement = document.querySelector('#pbf_data_json');
            if (destElement == null) {
                return
            }

            let config = JSON.parse(destElement.innerHTML)
            config['shipping'] = totals.shipping_incl_tax;

            var grandTotalSegment = totals['total_segments'].filter(function (segment) {
                return segment.code.indexOf('grand_total') !== -1;
            });

            var grandTotal = grandTotalSegment.length ? grandTotalSegment[0].value : 0;

            var pbfTotalSegment = totals['total_segments'].filter(function (segment) {
                return segment.code.indexOf('pbf_finance_amount') !== -1;
            });

            var pbfTotal = pbfTotalSegment.length ? pbfTotalSegment[0].value : 0;

            if (grandTotal && pbfTotal) {
                config['total'] = grandTotal + Math.abs(pbfTotal);
            } else {
                config['total'] = totals.shipping_incl_tax + totals.subtotal_incl_tax + totals.discount_amount;
            }

            destElement.innerHTML = JSON.stringify(config);
        };

        /**
         * Hook to KnockoutJS observer for change on totals JS instance
         */
        quote.totals.subscribe(function (totals) {
            if (self.financeEnabled()) {
                self.addTotalsWarning();
            }

            self.updateCarTotalsJson(totals, self.financeEnabled());

            //Wait for totals will be added to document to add additional info to it.
            observer.observe(document.body, {childList: true, subtree: true});

            //Wait for payment methods added to disable address editing
            payments_observer.observe(document.body, {childList: true, subtree: true});
        });


        if (self.financeEnabled()) {
            setTimeout(function () {
                self.hideNonUKAddresses();
                self.resetShippingAddress();
            }, 3000);
        }

        return self;
    };

});
