<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Block\Adminhtml\Installation;

use Magento\Backend\Block\Template as BlockTemplate;
use Magento\Backend\Block\Template\Context as BlockContext;
use Magento\Framework\App\Area;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class Websocket extends BlockTemplate
{
    private StoreConfigProvider $storeConfigProvider;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private StoreManagerInterface $storeManager;
    private \Magento\Framework\App\State $appState;

    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        StoreConfigProvider $storeConfigProvider,
        BlockContext $context,
        StoreManagerInterface $storeManager,
        \Magento\Framework\App\State $appState,
        array $data = []
    ) {
        parent::__construct($context, $data);
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->storeManager = $storeManager;
        $this->appState = $appState;
    }

    /**
     * @return bool
     */
    public function isPbfEnabled(): bool
    {
        list($scope, $scopeId) = $this->pbfWidgetConfigProvider->getInstallationPoint();
        
        return $this->storeConfigProvider->isPayByFinanceEnabled($scope, $scopeId);
    }

    public function getPbfInstallHmac(): ?string
    {
        list($scope, $scopeId) = $this->pbfWidgetConfigProvider->getInstallationPoint();
        return $this->storeConfigProvider->getPbfInstallHmac($scope, $scopeId);
    }

    public function getCanApplyFromCart()
    {
        return (bool)$this->pbfWidgetConfigProvider->canApplyFromCart();
    }

    public function getReturnUrl()
    {
        list($scopeType, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();
        $params = [$scopeType => $scopeId, StoreConfigProvider::DELETE_INSTALL_HMAC_PARAM => 1];
        return $this->getUrl('*/*/', $params);
    }

    public function getShopDomainHost(): string
    {
        return $this->pbfWidgetConfigProvider->getShopDomain();
    }

    public function getDefaultStoreShopDomainHost(): string
    {
        return $this->pbfWidgetConfigProvider->getDefaultStoreShopDomainHost();
    }

    public function getStoreCode()
    {
        $pbfInstallScope = $this->_scopeConfig->getValue(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE);
        //When on admin, it may not been saved yet:
        if ($pbfInstallScope == null && $this->appState->getAreaCode() == Area::AREA_ADMINHTML) {
            list($scopeType, $scopeId) = $this->pbfWidgetConfigProvider->getAdminReqScopeAndId();
            $pbfInstallScope = $scopeType;
        }

        //When PBF is installed on "for all" level:
        if ($pbfInstallScope == ScopeConfigInterface::SCOPE_TYPE_DEFAULT) {
            return "all";
        }

        $store = $this->pbfWidgetConfigProvider->getShopStore();
        if ($store == null) {
            throw new \Exception("Cannot find store view for scope.");
        }

        return $store->getCode();
    }

    public function getPbfServerVersion(): string
    {
        return $this->pbfWidgetConfigProvider->getPbfServerVersion();
    }

    /**
     * @return string
     */
    public function getWebSocketUrl(): string
    {
        $url = str_replace(
            'https://',
            'wss://',
            $this->pbfWidgetConfigProvider->getAppServerUrl() . PbfWidgetConfigProvider::INSTALLATION_WEBSOCKET_URL
        );

        return str_replace('http://', 'ws://', $url) . '?sv=' . $this->getPbfServerVersion();
    }
}
