<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Controller\Adminhtml\Configuration;

use Exception;
use Magento\Backend\App\Action\Context;
use Magento\Cms\Model\ResourceModel\Page\CollectionFactory as PageCollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Escaper;
use Magento\Framework\Exception\AuthenticationException;
use Magento\Framework\Exception\State\UserLockedException;
use Magento\Integration\Api\IntegrationServiceInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoresConfig;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class UninstallPost extends Base
{
    const ADMIN_RESOURCE = 'Novuna_Pbf::paybyfinance';
    private PbfLoggerInterface $logger;
    private Escaper $escaper;
    private IntegrationServiceInterface $integrationService;
    private StoreConfigProvider $storeConfigProvider;
    private PageCollectionFactory $pageCollectionFactory;
    private PbfWidgetConfigProvider $pbfWidgetConfigProvider;
    private StoresConfig $storesConfig;
    private WriterInterface $writerInterface;

    public function __construct(
        StoreConfigProvider $storeConfigProvider,
        IntegrationServiceInterface $integrationService,
        Escaper $escaper,
        PbfLoggerInterface $logger,
        PageCollectionFactory $pageCollectionFactory,
        Context $context,
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        StoresConfig $storesConfig,
        WriterInterface $writerInterface
    ) {
        $this->integrationService = $integrationService;
        $this->escaper = $escaper;
        $this->logger = $logger;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->pageCollectionFactory = $pageCollectionFactory;
        $this->pbfWidgetConfigProvider = $pbfWidgetConfigProvider;
        $this->storesConfig = $storesConfig;
        $this->writerInterface = $writerInterface;
        parent::__construct($context);
    }

    public function execute()
    {
        try {
            if ($this->isValidPostRequest()) {
                $this->validateUser();

                list($scope, $scopeId) = $this->pbfWidgetConfigProvider->getInstallationPoint();
                $dbScope = $this->getDBScope($scope);
                //Do not use "save", as it will leave 0 for that level, breaking future installs on upper levels:
                $this->writerInterface->delete(StoreConfigProvider::XML_PATH_PAYBYFINANCE_ENABLED, $dbScope, $scopeId);

                $this->deletePbfConfigSettings($dbScope, $scopeId);

                if ($this->getPBFInstalledOnStoreviews() == 0) {
                    $this->deleteIntegration();
                    $this->deletePbfCmsPages($dbScope, $scopeId);
                    $this->writerInterface->delete(StoreConfigProvider::XML_PATH_PAYBYFINANCE_INSTALL_SCOPE);
                }
            }
        } catch (UserLockedException $e) {
            $this->_auth->logout();
            $this->_redirect('*');
        } catch (AuthenticationException $e) {
            $this->messageManager->addErrorMessage($e->getMessage());
        } catch (\Exception $e) {
            $this->logger->critical($e);
            $this->messageManager->addErrorMessage($this->escaper->escapeHtml($e->getMessage()));
        }
        $this->redirect();
    }

    public function getPBFInstalledOnStoreviews()
    {
        $enbldStoreViews = $this->storesConfig->getStoresConfigByPath(StoreConfigProvider::XML_PATH_PAYBYFINANCE_ENABLED);
        $cnt = 0;
        foreach ($enbldStoreViews as $enabled) {
            if ((bool)$enabled) {
                $cnt++;
            }
        }
        return $cnt;
    }

    /**
     * @throws \Magento\Framework\Exception\LocalizedException
     */
    private function deletePbfConfigSettings($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT, $scopeId = 0)
    {
        $pbfPaths = [
            StoreConfigProvider::XML_PATH_PBF_GENERAL_SETTINGS,
            StoreConfigProvider::XML_PATH_PBF_INSTALL_HMAC,
            StoreConfigProvider::XML_PATH_PBF_WIDGET_SCRIPT_SRC,
            StoreConfigProvider::XML_PATH_PBF_WIDGET_SCRIPT_SRC . '_OnPDP+BasketLoad',
            StoreConfigProvider::XML_PATH_PBF_WIDGET_SCRIPT_SRC . '_OnCheckoutSuccess'
        ];
        $this->storeConfigProvider->deleteConfigByPaths($pbfPaths, $scope, $scopeId);
    }

    //TODO - manage CMS pages under correct storeview:
    private function deletePbfCmsPages()
    {
        $collection = $this->pageCollectionFactory->create();
        $collection->addFieldToFilter(
            'identifier',
            ['like' => 'pbf_managed_%']
        );
        foreach ($collection as $page) {
            $page->delete();
        }
    }

    private function deleteIntegration(): bool
    {
        $integrationName = StoreConfigProvider::PBF_INTEGRATION_IDENTIFIER;
        $integration = $this->integrationService->findByName($integrationName);
        if ($integration && $integration->getId()) {
            $this->integrationService->delete($integration->getId());
            $this->messageManager->addSuccessMessage(__('PBF Integration deleted'));
            return true;
        }
        return false;
    }

    private function getDBScope($scope)
    {
        switch ($scope) {
            case ScopeInterface::SCOPE_WEBSITE : return ScopeInterface::SCOPE_WEBSITES;
            case ScopeInterface::SCOPE_STORE : return ScopeInterface::SCOPE_STORES;
            default: return $scope;
        }
    }
}
