<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Controller\ApplyPbf;

use Laminas\Uri\Uri;
use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Framework\App\Request\Http;
use Magento\Framework\Controller\Result\RawFactory;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Framework\App\Request\InvalidRequestException;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\View\Result\PageFactory;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\App\CsrfAwareActionInterface;
use Magento\Framework\App\Action\HttpPostActionInterface;
use Magento\Framework\Controller\ResultInterface;
use Novuna\Pbf\Controller\PbfWidget;
use Novuna\Pbf\Model\Widget\PbfWidgetConfigProvider;

class Index extends PbfWidget implements HttpPostActionInterface, CsrfAwareActionInterface
{
    protected CheckoutSession $checkoutSession;

    public function __construct(
        PbfWidgetConfigProvider $pbfWidgetConfigProvider,
        PageFactory $resultPageFactory,
        Curl $curl,
        Http $request,
        RawFactory $resultRawFactory,
        StoreManagerInterface $storeManager,
        Uri $uri,
        CheckoutSession $checkoutSession
    ) {
        parent::__construct(
            $pbfWidgetConfigProvider,
            $resultPageFactory,
            $curl,
            $request,
            $resultRawFactory,
            $storeManager,
            $uri
        );

        $this->checkoutSession = $checkoutSession;
    }

    /**
     * Execute view action
     * @return ResultInterface
     * @throws NoSuchEntityException
     * @todo  add signature to requests
     */
    public function execute(): ResultInterface
    {
        $proxiedPath = PbfWidgetConfigProvider::PROXIED_PATH;
        $destinationPath = PbfWidgetConfigProvider::PROXY_DESTINATION_PATH;

        $uri = str_replace($proxiedPath, '', $this->request->getRequestUri());
        $parsedUri = $this->uri->parse($uri);
        $path = $parsedUri->getPath();
        $params = $parsedUri->getQueryAsArray();
        //shop param is passed from PBF widget

        $url = $this->pbfWidgetConfigProvider->getAppServerUrl() . $destinationPath . $path . "?" . http_build_query($params);

        $content = json_decode($this->request->getContent(), true);

        $quote = $this->checkoutSession->getQuote();
        $billingAddress = $quote->getBillingAddress();

        $content['checkoutToken'] = $this->checkoutSession->getQuoteId();
        $content['billingAddress'] = $this->remapAddressFields($billingAddress->getData());
        $content['billingAddress']['Address'] = $billingAddress->getStreet();
        $content['postcode'] = $billingAddress->getPostcode();

        $this->curl->post($url, json_encode($content));

        $result = $this->resultRawFactory->create();
        foreach ($this->curl->getHeaders() as $k => $v) {
            if ($k != "transfer-encoding" && $v != "chunked") {
                $result->setHeader($k, $v);
            }
        }
        $result->setHttpResponseCode($this->curl->getStatus());
        return $result->setContents($this->curl->getBody());
    }

    private function remapAddressFields($address)
    {
        $fieldMap = [
            "street" => "Street",
            "telephone" => "Phone",
            "postcode" => "PostCode",
            "city" => "Town",
            "firstname" => "FirstName",
            "lastname" => "SurName",
        ];

        $remapped = [];
        foreach ($address as $k => $v) {
            $remapped[$fieldMap[$k] ?? $k] = $v;
        }

        return $remapped;
    }
}
