<?php

/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */

declare(strict_types=1);

namespace Novuna\Pbf\Model\Api;

use Magento\Framework\Webapi\Exception as WebapiException;
use Magento\Quote\Api\CartRepositoryInterface as QuoteRepository;
use Magento\Sales\Model\Order\Status\HistoryFactory;
use Novuna\Pbf\Api\Data\CommentWrapperInterface;
use Novuna\Pbf\Api\Data\DraftOrderWrapperInterface;
use Novuna\Pbf\Api\DraftOrderInterface;
use Novuna\Pbf\Model\Data\CommentWrapper;
use Novuna\Pbf\Model\Data\DraftOrderWrapper;
use Novuna\Pbf\Model\DraftOrder\DraftOrderMaker;
use Novuna\Pbf\Model\DraftOrderRepository;
use Novuna\Pbf\Model\DraftOrder\Quote\QuoteConverter;
use Novuna\Pbf\Model\StoreConfigProvider;

class DraftOrderService implements DraftOrderInterface
{
    private StoreConfigProvider $storeConfigProvider;
    private DraftOrderMaker $draftOrderMaker;
    private QuoteConverter $quoteConverter;
    private QuoteRepository $quoteRepository;
    private DraftOrderRepository $draftOrderRepository;

    public function __construct(
        DraftOrderMaker $draftOrderMaker,
        QuoteConverter $quoteConverter,
        StoreConfigProvider $storeConfigProvider,
        QuoteRepository $quoteRepository,
        DraftOrderRepository $draftOrderRepository
    ) {
        $this->draftOrderMaker = $draftOrderMaker;
        $this->quoteConverter = $quoteConverter;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->quoteRepository = $quoteRepository;
        $this->draftOrderRepository = $draftOrderRepository;
    }

    public function createDraftOrder(string $quoteIdMasked, float $shippingPrice, bool $pbfTakingDeposit): ?DraftOrderWrapperInterface
    {
        $draftOrder = $this->draftOrderMaker->makeDraftOrder($quoteIdMasked, $pbfTakingDeposit);

        $quote = $draftOrder->getQuote();
        $shippingAddress = $quote->getShippingAddress();
        $shippingAddress->setShippingInclTax($shippingPrice);
        $quote->reserveOrderId();
        $this->quoteRepository->save($quote);

        return new DraftOrderWrapper($draftOrder);
    }

    public function getDraftOrder(string $quoteIdMasked): ?DraftOrderWrapperInterface
    {
        $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);
        return new DraftOrderWrapper($draftOrder);
    }

    public function reserveOrderId(string $quoteIdMasked): ?string
    {
        $quote = $this->quoteRepository->get($quoteIdMasked);
        $reservedId = $quote->reserveOrderId()->getReservedOrderId();
        $this->quoteRepository->save($quote);
        return $reservedId;
    }

    public function getByOrder(int $orderId): ?DraftOrderWrapperInterface
    {
        $draftOrder = $this->draftOrderMaker->getDraftOrderByOrderId($orderId);
        return new DraftOrderWrapper($draftOrder);
    }

    public function convertDraftOrder(string $quoteIdMasked): ?DraftOrderWrapperInterface
    {
        $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);
        if ($draftOrder->getId() && $draftOrder->getQuote()) {
            if ($draftOrder->getOrderId()) {
                throw new WebapiException(__('Draft order already converted'));
            }
            $order = $this->quoteConverter->convertDraftOrderToRealOrder($draftOrder->getQuote());
            $draftOrder->setOrder($order);
        } else {
            throw new WebapiException(__('Unable to find related draft order'));
        }
        return new DraftOrderWrapper($draftOrder);
    }

    public function updateAddresses(
        string $quoteIdMasked,
        \Magento\Quote\Api\Data\AddressInterface $orderAddress
    ): bool {

        $draftOrder = $this->draftOrderMaker->getDraftOrder($quoteIdMasked);
        if (!$draftOrder->getId()) {
            throw new WebapiException(__('Unable to find related draft order'));
        }

        $quote = $draftOrder->getQuote();

        $draftOrder->setAddress($orderAddress);

        $existingBilling = $quote->getBillingAddress();
        $existingBilling->addData($orderAddress->getData());
        $existingBilling->setCountryId("GB");
        $quote->setBillingAddress($existingBilling);

        $existingShipping = $quote->getShippingAddress();
        $existingShipping->addData($orderAddress->getData());
        $existingShipping->setCountryId("GB");
        $quote->setShippingAddress($existingShipping);

        //not sure why M2 is not filling that y default when it's required on converting to order:
        $quote->setCustomerEmail($quote->getBillingAddress()->getEmail());

        $this->quoteRepository->save($quote);
        $this->draftOrderRepository->save($draftOrder);

        return true;
    }

    /**
     * Adds a comment to a specified draft order.
     *
     * @param int $id The quote ID.
     * @param \Magento\Sales\Api\Data\OrderStatusHistoryInterface[] $statusHistoryItems Status history comment.
     * @return bool
     */
    public function addComment($id, $statusHistoryItems)
    {
        $quote = $this->quoteRepository->get($id);
        $json_comments = $quote->getPbfDraftComments();
        if ($json_comments == null) {
            $json_comments = "[]";
        }
        $comments = json_decode($json_comments);
        $lastId = $this->getLastCommentId($comments);

        foreach ($statusHistoryItems as $statusHistory) {
            $lastId++;
            $comment = new \stdClass();
            $comment->entity_id = $lastId;
            $comment->parent_id = $id;
            $comment->comment = $statusHistory->getComment();

            if ($statusHistory->getCreatedAt() == null) {
                $createdInLocal = date("Y-m-d H:i:s");
            } else {
                $created = strtotime($statusHistory->getCreatedAt() . ' UTC');
                $createdInLocal = date("Y-m-d H:i:s", $created);
            }
            $comment->created_at = $createdInLocal;
            $comments[] = $comment;
        }

        $quote->setPbfDraftComments(json_encode($comments));

        $this->quoteRepository->save($quote);
    }

    /**
     * Get a comments of a specified draft order.
     *
     * @param int $id The quote ID.
     * @return CommentWrapperInterface
     */
    public function getComments($id)
    {
        //This should be implemented here and quoteConverter should call it - but M2 does not allow two ways references:
        return new CommentWrapper($this->quoteConverter->getPbfDraftComments($id));
    }

    private function getLastCommentId($comments)
    {
        $lastId = 0;
        foreach ($comments as $c) {
            if ($c->entity_id > $lastId) {
                $lastId = $c->entity_id;
            }
        }
        return $lastId;
    }
}
