<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Model\Total\Quote;

use Magento\Checkout\Model\Session as CheckoutSession;
use Magento\Quote\Api\Data\ShippingAssignmentInterface as ShippingAssignment;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\Quote\Address\Total;
use Magento\Quote\Model\Quote\Address\Total\AbstractTotal;
use Magento\SalesRule\Model\Quote\Address\Total\ShippingDiscount;
use Magento\SalesRule\Model\Quote\Discount as DiscountCollector;
use Magento\SalesRule\Model\Validator;
use Novuna\Pbf\Model\StoreConfigProvider;

class FinanceAmount extends AbstractTotal
{
    const PAYBYFINANCE_TOTAL_CODE = 'pbf_finance_amount'; //key in totals
    private StoreConfigProvider $storeConfigProvider;
    private Validator $calculator;
    private CheckoutSession $checkoutSession;

    public function __construct(
        CheckoutSession $checkoutSession,
        Validator $calculator,
        StoreConfigProvider $storeConfigProvider
    ) {
        $this->checkoutSession = $checkoutSession;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->calculator = $calculator;
        $this->setCode(self::PAYBYFINANCE_TOTAL_CODE);
    }

    /**
     * @inheritdoc
     *
     * @param Quote $quote
     * @param ShippingAssignment $shippingAssignment
     * @param Total $total
     * @return ShippingDiscount
     */
    public function collect(Quote $quote, ShippingAssignment $shippingAssignment, Total $total): self
    {
        parent::collect($quote, $shippingAssignment, $total);
        /**
         * We have to remove PBF if coupon code applied and the widget will re-add it
         * pbf_coupon_conflict_resolve_plugin plugin will set discount amount before coupon code
         * if current discount != before discount then we would assume new coupon code added and that will change totals
         * If totals changed, PBF widget will re-send new PBF totals
         */
        $shippingAddress = $quote->getShippingAddress();
        if ($quote->getDiscountBeforeCouponApply() &&
            $quote->getDiscountBeforeCouponApply() != $shippingAddress->getDiscountAmount()) {
            $quote->setPbfFinanceAmount(0);
        }
        $address = $shippingAssignment->getShipping()->getAddress();
        $financeAmount = abs(floatval($quote->getPbfFinanceAmount())) ?? 0;
        $financeAmountExclTax = abs(floatval($quote->getPbfFinanceExclTaxAmount())) ?? 0;
        $items = $shippingAssignment->getItems();
        if (!count($items)) {
            return $this;
        }

        $total->addTotalAmount(self::PAYBYFINANCE_TOTAL_CODE, -$financeAmount);
        $total->addBaseTotalAmount(
            self::PAYBYFINANCE_TOTAL_CODE,
            -$financeAmount
        );

        $quote->setPbfFinanceAmount(-$financeAmount);

        $address->setPbfFinanceAmount($total->getPbfFinanceAmount());
        $address->setPbfBaseFinanceAmount($total->getPbfBaseFinanceAmount());

        return $this;
    }

    /**
     * @param Quote $quote
     * @param Total $total
     * @return array|null
     *
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     *
     */
    public function fetch(Quote $quote, Total $total)
    {
        if ($quote->getPbfFinanceAmount()) {
            return [
                'code' => $this->getCode(),
                'title' => __('Financed Amount'),
                'value' => -abs(floatval($quote->getPbfFinanceAmount()))
            ];
        }

        return null;
    }
}
