<?php
/**
 * Copyright (C) 2022 Novuna Consumer Finance
 * All rights reserved. See LICENCE.pdf for details
 */
declare(strict_types=1);

namespace Novuna\Pbf\Observer\Checkout;

use Magento\Framework\Event\Observer;
use Magento\Framework\Event\ObserverInterface;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Sales\Model\Order;
use Magento\Framework\Serialize\Serializer\Json;
use Novuna\Pbf\Api\SalesOrderMetaRepositoryInterface;
use Novuna\Pbf\Logger\PbfLoggerInterface;
use Novuna\Pbf\Model\StoreConfigProvider;
use Novuna\Pbf\Model\Data\OrderMetaFactory;

class SalesOrderSaveAfter implements ObserverInterface
{

    private PbfLoggerInterface $logger;
    private StoreConfigProvider $storeConfigProvider;
    private SalesOrderMetaRepositoryInterface $orderMetaRepository;
    private OrderMetaFactory $orderMetaFactory;
    const ORDER_CREATE_META_KEY = 'pbf_usedFinance';
    const ORDER_FINANCE_STATE_META_KEY = 'pbf_financeAppState';
    private Json $serializer;

    public function __construct(
        OrderMetaFactory $orderMetaFactory,
        SalesOrderMetaRepositoryInterface $orderMetaRepository,
        PbfLoggerInterface $logger,
        StoreConfigProvider $storeConfigProvider,
        Json $serializer
    ) {
        $this->orderMetaRepository = $orderMetaRepository;
        $this->orderMetaFactory = $orderMetaFactory;
        $this->logger = $logger;
        $this->storeConfigProvider = $storeConfigProvider;
        $this->serializer = $serializer;
    }

    public function execute(Observer $observer)
    {
        $order = $observer->getOrder();
        //PbfAdditionalData is set from QuoteSubmitBefore, so this will be skipped from other order save
        if ($order->getId() && $order->getPbfFinanceEnabled() && $order->getPbfAdditionalData()) {
            $orderId = (int)$order->getId();
            $quoteId = (int)$order->getQuoteId();

            $orderMetaModel = $this->getOrderMetaSaveModel(
                $quoteId,
                self::ORDER_CREATE_META_KEY,
                'quote_id'
            );
            if ($order instanceof Order) {
                $orderMetaModel->setOrderId($orderId);
            }
            if (!$orderMetaModel->getId()) {
                $orderMetaModel->setQuoteId($order->getQuoteId());
                $orderMetaModel->setKey(self::ORDER_CREATE_META_KEY);
                $orderMetaModel->setValue($order->getPbfAdditionalData());
            }

            $pbfData = $this->serializer->unserialize((string)$order->getPbfAdditionalData());
            $order->unsetData('pbf_additional_data');

            $pbfAppState = $this->getOrderMetaSaveModel(
                $quoteId,
                self::ORDER_FINANCE_STATE_META_KEY,
                'quote_id'
            );
            if ($order instanceof Order) {
                $pbfAppState->setOrderId($order->getId());
            }
            if (!$pbfAppState->getId()) {
                $pbfAppState->setQuoteId($order->getQuoteId());
                $pbfAppState->setKey(self::ORDER_FINANCE_STATE_META_KEY);

                //Widget is sending initial value of pbfAppState metafield:
                $pbfAppState->setValue(
                    $this->serializer->serialize($pbfData['versions'])
                );
            }
            try {
                $this->orderMetaRepository->save($orderMetaModel);
                $this->orderMetaRepository->save($pbfAppState);
            } catch (LocalizedException $e) {
                $this->logger->error($e->getMessage());
            }
        }
    }

    private function getOrderMetaSaveModel(int $id, $metaKey, $field)
    {
        try {
            $saveModel = $this->orderMetaRepository->getByMetaFieldKey(
                $id,
                $metaKey,
                $field
            );
        } catch (NoSuchEntityException $e) {
            $saveModel = $this->orderMetaFactory->create(); //create new meta field if it is not exist
        }
        return $saveModel;
    }
}
